#!/usr/bin/env python
import sys
import argparse
import subprocess

# Simple python wrapper for check_postgres.pl. Allows for aggregating relevant check_postgres.pl pgbouncer checks.
# Returns the status of the highest exit code seen (0,1,2). UNKNOWN (3) is returned only when script catches an exception.
# Assumes run as nagios user (as nrpe executes passive checks as the nagios user)

# Error exit codes
OK       = (0, 'OK')
WARN     = (1, 'WARN')
CRITICAL = (2, 'CRITICAL')
UNKNOWN  = (3, 'UNKNOWN')

CHECK = '/usr/local/monitor_scripts/check_postgres.pl' # executable for subprocess.Popen()
CHECK_TIMEOUT = '3'  # max time we should allow a pgbouncer check to return values.
status_array = [] # array of return output(s) [(<return_code>, <return_message>), ..]

# Parse arguments
parser = argparse.ArgumentParser()
parser.add_argument('-v', '--verbose', action='store_true', default=False, help='Print detailed output to stdout.')
parser.add_argument('-c', '--check', default='PG', help='check_postgres specific action')
parser.add_argument('-u', '--dbuser', default='nagios', help='Postgres username')
parser.add_argument('-p', '--dbport', default= 5432, help='Postgres port')
options = parser.parse_args()

# Exit code function
def exit_with(err_lvl, mesg):
    print '%s: %s' % (err_lvl[1], mesg)
    sys.exit(err_lvl[0])


# Check postgres checks (assumes .pgpass file exists in user's homedir. In the case of nagios, it's /var/lib/nagios/.pgpass)
def template_popen(action, dbport):
    '''
        Template for calling a subprocess. Returns (stdout, stderr) of subprocess call
    '''
    pipe = None if options.verbose else subprocess.PIPE
    proc = subprocess.Popen([CHECK, '--action', action, '-u', options.dbuser, '--port', str(dbport), '-t', CHECK_TIMEOUT, '--host', 'localhost'], stderr=pipe, stdout=pipe)
    return proc.communicate()

def check_pgb_pool_cl_active(dbport):
    status_array.append(template_popen('pgb_pool_cl_active', dbport))

def check_pgb_pool_cl_waiting(dbport):
    status_array.append(template_popen('pgb_pool_cl_waiting', dbport))

def check_pgb_pool_maxwait(dbport):
    status_array.append(template_popen('pgb_pool_maxwait', dbport))

def check_pgb_pool_sv_active(dbport):
    status_array.append(template_popen('pgb_pool_sv_active', dbport))

def check_pgb_pool_sv_idle(dbport):
    status_array.append(template_popen('pgb_pool_sv_idle', dbport))

def check_pgb_pool_sv_login(dbport):
    status_array.append(template_popen('pgb_pool_sv_login', dbport))

def check_pgb_pool_sv_tested(dbport):
    status_array.append(template_popen('pgb_pool_sv_tested', dbport))

def check_pgb_pool_sv_used(dbport):
    status_array.append(template_popen('pgb_pool_sv_used', dbport))

def check_pgbouncer_backends(dbport):
    status_array.append(template_popen('pgbouncer_backends', dbport))

def check_backends(dbport):
    status_array.append(template_popen('backends', dbport))

def check_hitratio(dbport):
    status_array.append(template_popen('hitratio', dbport))


def run_checks(check, dbport):
    '''
        Reads options taken and runs the check, populates status array
    '''
    check_dict = {
        'pgb_pool_cl_active': check_pgb_pool_cl_active,
        'pgb_pool_cl_waiting': check_pgb_pool_cl_waiting,
        'pgb_pool_maxwait': check_pgb_pool_maxwait,
        'pgb_pool_sv_active': check_pgb_pool_sv_active,
        'pgb_pool_sv_idle': check_pgb_pool_sv_idle,
        'pgb_pool_sv_login': check_pgb_pool_sv_login,
        'pgb_pool_sv_tested': check_pgb_pool_sv_tested,
        'pgb_pool_sv_used': check_pgb_pool_sv_used,
        'pgbouncer_backends': check_pgbouncer_backends,
        'backends' : check_backends,
        'hitratio': check_hitratio
    }

    if check == 'PGB': # pgbouncer checks all have the str "pgb" in it somewhere
        for check_name, check_func in check_dict.items():
            if "pgb" in check_name:
                check_func(dbport)
    elif check == 'PG': # postgres checks don't have the str "pgb" in it
        for check_name, check_func in check_dict.items():
            if "pgb" not in check_name:
                check_func(dbport)
    else: # otherwise, give a specific action
        if check not in check_dict:  # Return UNKNOWN if misconfigured check
            exit_with(UNKNOWN, "check_postgres check '{}' is not a valid option".format(check))
        else:
            return check_dict[check](dbport) # Otherwise, carry out the specific action


def validate_check_health(check_opt):
    '''
        Goes through status_array and aggregates return messages and exits accordingly.
    '''
    health = {'OK': 0, 'WARN': 0, 'CRIT': 0, 'UNKNOWN': 0} # dict of check result organized by health
    status_str = ""
    check_return_status = OK
    for check in status_array: # Loop through each check result and grep for check_postgres.pl script output
        if 'OK' in check[0]:
            health['OK'] += 1
        elif 'WARN' in check[0]:
            health['WARN'] += 1
            if check_return_status[0] < 1:
                check_return_status = WARN
        elif 'CRIT' in check[0]:
            health['CRIT'] += 1
            if check_return_status[0] < 2:
                check_return_status = CRITICAL
        elif 'UNKNOWN' in check[0]:
            health['UNKNOWN'] += 1
            if check_return_status[0] == 0: # only change to UNKNOWN if previous state was OK.
                check_return_status = UNKNOWN

        if 'OK' not in check[0]: # only append non-OK outputs for debugging.
            status_str += '{}\n'.format(check[0])
    # format string for readability
    fmt_str = 'Postgres'
    if check_opt == 'PGB':
        fmt_str = 'PGBouncer'
    formatted_check_str = 'check_postgres_health: {0} is {1}\n{2} checks run: {3} OK, {4} WARN, {5} CRIT.\n{6}'.format(
        fmt_str, check_return_status[1], len(status_array), health['OK'], health['WARN'], health['CRIT'], status_str
    )
    exit_with(check_return_status, formatted_check_str)


#
# Main execution --> Do checks, return health
#
if __name__ == '__main__':
    run_checks(options.check, options.dbport)
    validate_check_health(options.check)
