#!/usr/bin/env python
import requests
import argparse
import json
import sys

# A quick script to replace the functionality of gamekeeper and other rabbitmq checks
# Queries its own HTTP API to get relevant metrics.

# Error exit codes
OK = (0, 'OK')
WARN = (1, 'WARN')
CRITICAL = (2, 'CRITICAL')
UNKNOWN = (3, 'UNKNOWN')

# Parse arguments
parser = argparse.ArgumentParser()
parser.add_argument('--host', help="host FQDN")
parser.add_argument('--check', default="memory", help="One of three types of checks: memory, queue, or partition")
parser.add_argument('-q', '--queue', help="a queue name to check if --check options is 'queue'")
parser.add_argument('-s', '--state', default="all", help="state of messages in 'queue' to monitor", choices=['ready', 'unack', 'all'])
parser.add_argument('-w', '--warn', default=1, help="Generic WARN threshold")
parser.add_argument('-c', '--crit', default=1, help="Generic CRIT threshold")
options = parser.parse_args()

# User must provide a queue name if --check's value is "queue"
if options.check.lower() == "queue" and options.queue is None:
    parser.error("'--check queue' requires -q (--queue)")

# Exit code function
def exit_with(err_lvl, mesg):
    print '%s: %s' % (err_lvl[1], mesg)
    sys.exit(err_lvl[0])

# Takes a generic metric value and compares it to the
def alert_logic(metric, check):
    sanitized = options.check.lower()
    if metric is None:
        exit_with(UNKNOWN, "Failed to garner metric. Please investigate.")
    if metric >= float(options.crit):
        exit_with(CRITICAL, "{} is CRITICAL! Value is {}".format(sanitized, metric))
    elif metric >= float(options.warn):
        exit_with(WARN, "{} is WARNING. Value is {}".format(sanitized, metric))
    exit_with(OK, "{} is OK. Value is {}".format(sanitized, metric))

def form_endpoint_template():
    sanitized = options.check.lower()
    if sanitized == "memory":
        return "http://127.0.0.1:15672/api/nodes/rabbit@{}/memory".format(options.host)
    elif sanitized == "queue":
        return "http://127.0.0.1:15672/api/queues"
    elif sanitized == "partition":
        return "http://127.0.0.1:15672/api/nodes/rabbit@{}".format(options.host)
    else:
        exit_with(UNKNOWN, "Illegal check option {}".format(sanitized))


def parse_endpoint_response(r):
    # NOTE: no "else" at the end like in form_endpoint_template because
    # form_endpoint_template is always called before parse_endpoint_response
    sanitized = options.check.lower()
    # Memory check uses the rabbitmq calculated rss value.
    # See https://www.rabbitmq.com/memory-use.html for more details
    if sanitized == "memory":
        return json.loads(r.text)['memory']['total']['rss']/float(1000000000)
    # Checks the total of 'messages_ready' and 'messages_unacknowledged' value of a specific queue to know if there is queue backup or crashes in the application layer
    elif sanitized == "queue":
        for queue in json.loads(r.text):
            if options.queue.lower() == queue['name']:
                if options.state.lower() == "ready":
                    return (queue['messages_ready'])
                elif options.state.lower() == "unack":
                    return (queue['messages_unacknowledged'])
                if options.state.lower() == "all":
                    return (queue['messages_ready'] + queue['messages_unacknowledged'])
        exit_with(CRITICAL, "Queue {} does not exist! Are you sure this is what you want?".format(options.queue.lower()))
    # Queries the cluster to see if any network partitions exist
    elif sanitized == "partition":
        return len(json.loads(r.text)['partitions'])


def check():
    # Get info from rabbit HTTP API
    endpoint_template = form_endpoint_template()
    try:
        r = requests.get(endpoint_template.format(options.host), auth=('guest', 'guest'))
    except requests.ConnectionError as e:
        exit_with(CRITICAL, "Failed to connect to rabbit management UI: {}".format(e))
    except Exception as e:
        exit_with(UNKNOWN, "Failed to Query health endpoint: {}".format(e))
    # parse output from rabbit
    try:
        metric = parse_endpoint_response(r)
    except Exception as e:
        exit_with(UNKNOWN, "Failed to decode rabbit health response: {}".format(e))
    # handle alerting logic
    alert_logic(metric, check)

if __name__ == "__main__":
    check()
