#!/usr/bin/env python3

import argparse
import json
import logging
import subprocess
import sys

RETURN = {0: 'OK', 1: 'WARN', 2: 'CRIT', 3: 'UNKNOWN'}
DEFAULT_IFNAME = 'team0'


def get_state(ifname):
    try:
        cmd = subprocess.run(['sudo',
                              '/usr/bin/teamdctl',
                              ifname,
                              'state',
                              'dump'],
                             check=True,
                             stdout=subprocess.PIPE,
                             stderr=subprocess.PIPE
                             )
    except (subprocess.SubprocessError, subprocess.CalledProcessError):
        logging.exception('UNKNOWN: teamdctl exec returned error:')
        sys.exit(3)
    except subprocess.TimeoutExpired:
        logging.exception('UNKNOWN: teamdctl exec timed out:')
        sys.exit(3)
    try:
        output = json.loads(cmd.stdout.decode())
    except (json.JSONDecodeError, ValueError, TypeError):
        logging.exception('UNKNOWN: failed to parse json from teamdctl:')
        sys.exit(3)

    return output


class Port(dict):
    def __init__(self, info):
        self.update(info)

    def __repr__(self):
        return "interface: {} link_up: {} lag_state: {}".format(self.interface_name(), self.link_status(), self.lacp_state())  # noqa: E501

    def interface_name(self):
        return self['ifinfo']['ifname']

    def link_status(self):
        return self['link']['up']

    def lacp_state(self):
        return self['runner']['state'] == 'current'

    def aggregator_id(self):
        return self['runner']['aggregator']['id']


if __name__ == '__main__':
    '''parse output of teamdctl and monitor interface state'''
    parser = argparse.ArgumentParser(description=__doc__)
    parser.add_argument('-v', '--verbose', action='store_true', default=False,
                        help='turn on verbose logging')
    parser.add_argument('-i', '--interface', default=DEFAULT_IFNAME,
                        help="check status of this interface. default: {}".format(DEFAULT_IFNAME))  # noqa: E501

    args = parser.parse_args()
    ports = list()
    aggregator = set()
    statuses = [(0, "interface {} up and running".format(args.interface))]

    if args.verbose:
        level = logging.DEBUG
    else:
        level = logging.INFO
    logging.basicConfig(level=level, format='%(message)s')

    # iterate through ports looking for problems
    port_state = get_state(args.interface)['ports']
    for p in port_state:
        ports.append(Port(port_state[p]))

    for p in ports:
        if not p.link_status():
            statuses.append((2, "{} link down".format(p.interface_name())))  # noqa: E501
        if not p.lacp_state():
            statuses.append((2, "{} lacp down".format(p.interface_name())))  # noqa: E501

        aggregator.add(p.aggregator_id())

    if len(aggregator) > 1:
        statuses.append((2, "LACP misconfigured on switch or cabling issue"))

    if len(ports) < 2:
        statuses.append((2,
                         "{} has only one physical interface".format(args.interface)  # noqa: E501
                         ))

    # additional checks could go here looking at other data from get_state()

    # Walk through statuses from unknown, crit, warn to ok
    for k in sorted(RETURN, key=int, reverse=True):
        messages = list()
        for (s, msg) in statuses:
            if s == k:
                messages.append(msg)
        if messages:
            # print and exit on the most serious errors first
            print("{}: {}".format(RETURN[k], ', '.join(messages)))
            # extended info
            for p in ports:
                print(p)
            sys.exit(k)
