#!/usr/bin/env python2
'''Multiple Nagios check for varnish health based on varnishstat.'''
import argparse
import collections
import doctest
import itertools
import json
import nagiosplugin
import subprocess
import sys
import tempfile


SMA_USAGE_NAME = lambda: 'sma_usage'
NUKED_NAME = lambda: 'nuked_delta'


def default(option):
    '''str -> any
    Lookup the default value for an option
    '''
    defaults = dict(
       varnish_name='/dev/shm/varnish/',
       sma_usage_warn='0:0.9',
       sma_usage_crit='0:0.95',
       sma_name='s0',
       nuked_crit='0:0',
       check_statefile='/var/tmp/varnish_stats_check.state',
       verbosity=0
    )
    return defaults[option]


class Varnishstats(nagiosplugin.Resource):
    '''A nagiosplugin.Resource implementation to pull a variety of fields and
    emit multiple metrics while only making one call to varnishstat.
    '''

    def __init__(self, varnish_name, statefile_name, collectors):
        fields = set(itertools.chain.from_iterable(c.fields for c in collectors))
        self.__json_data = self._run(varnish_name, fields)
        self.__statefile = statefile_name
        self.__collectors = collectors
        self.__metrics = None

    @staticmethod
    def _command(name, fields):
        '''str, [str] -> [str]
        Return a subprocess command arg vector for the indicated varnish fields
        and varnish name.

        >>> Varnishstats._command('foo', ['a', 'b'])
        ['varnishstat', '-n', 'foo', '-j', '-f', 'a', '-f', 'b']
        '''
        return reduce(lambda acc, cur: acc + ['-f', cur], fields, ['varnishstat', '-n', name, '-j'])

    @staticmethod
    def _run(name, fields):
        '''str, [str] -> effect str
        Run varnishstat to fetch the indicated fields and parse the resulting
        json.
        '''
        return json.loads(subprocess.check_output(Varnishstats._command(name, fields)))

    def probe(self):
        '''Collect metrics from all of the collectors, threading the cookie
        through, and then save the metrics in case probe is called again.
        '''
        if self.__metrics is None:
            metrics = list()
            with nagiosplugin.Cookie(self.__statefile) as cookie:
                for c in self.__collectors:
                    metrics.extend(c.collect(self.__json_data, cookie))
            self.__metrics = metrics
        return self.__metrics


class CollectSMAUsage(object):
    '''A collector for use with the Varnishstats nagiosplugin.Resource which
    emits the memory usage, as a fraction on [0, 1], of the varnish instance.
    '''

    def __init__(self, sma_name):
        self.__sma_name = sma_name

    @property
    def _g_bytes(self):
        return 'SMA.%s.g_bytes' % self.__sma_name

    @property
    def _g_space(self):
        return 'SMA.%s.g_space' % self.__sma_name

    @property
    def fields(self):
        return [self._g_bytes, self._g_space]

    @staticmethod
    def _sma_usage(sma_g_bytes, sma_g_space):
        '''int, int -> float
        Compute usage from the varnish sma bytes used and bytes free.

        >>> CollectSMAUsage._sma_usage(1, 99)
        0.01
        '''
        return float(sma_g_bytes) / (sma_g_bytes + sma_g_space)

    def collect(self, json_data, _):
        '''parsed-json-data, nagiosplugin.Cookie -> [nagiosplugin.Metric]
        Produce sma usage metric from varnishstat json output.

        >>> CollectSMAUsage('foo').collect({
        ... 'SMA.foo.g_bytes':dict(value=2),
        ... 'SMA.foo.g_space':dict(value=98),
        ... },
        ... NotImplemented)
        [Metric(name='sma_usage', value=0.02, uom=None, min=0, max=1, context='sma_usage', contextobj=None, resource=None)]
        '''
        return [nagiosplugin.Metric(
            name=SMA_USAGE_NAME(),
            value=self._sma_usage(json_data[self._g_bytes]['value'], json_data[self._g_space]['value']),
            min=0,
            max=1,
        )]


# XXX: in the future, it might be valuable to define a "delta collector" rather than repeating the pattern used below to compute the delta
class CollectDeltaLRUNuked(object):
    '''A collector for use with the Varnishstats nagiosplugin.Resource which
    emits the delta between sequential pairs of lru_nuked values.
    '''

    def __init__(self):
        pass # only point of this method is to make the fields property work

    _nuked = 'MAIN.n_lru_nuked'

    @property
    def fields(self):
        return [self._nuked]

    @classmethod
    def collect(cls, json_data, state_cookie):
        '''parsed-json-data, nagiosplugin.Cookie -> effect [nagiosplugin.Metric]
        Produce the delta n_lru_nuked metric from the varnishstat json output
        and the previous value in the state cookie. Save the current value to
        the state cookie for the next call. If there is no value in the state
        cookie, no metric is produced by this call.

        Example: n_lru_nuked doesn't change
        >>> with tempfile.NamedTemporaryFile() as fd:
        ...     with nagiosplugin.Cookie(fd.name) as cookie:
        ...         CollectDeltaLRUNuked.collect({'MAIN.n_lru_nuked':dict(value=2)}, cookie)
        ...         CollectDeltaLRUNuked.collect({'MAIN.n_lru_nuked':dict(value=2)}, cookie)
        []
        [Metric(name='nuked_delta', value=0, uom='count', min=0, max=None, context='nuked_delta', contextobj=None, resource=None)]

        Example: n_lru_nuked increases
        >>> with tempfile.NamedTemporaryFile() as fd:
        ...     with nagiosplugin.Cookie(fd.name) as cookie:
        ...         CollectDeltaLRUNuked.collect({'MAIN.n_lru_nuked':dict(value=5)}, cookie)
        ...         CollectDeltaLRUNuked.collect({'MAIN.n_lru_nuked':dict(value=8)}, cookie)
        []
        [Metric(name='nuked_delta', value=3, uom='count', min=0, max=None, context='nuked_delta', contextobj=None, resource=None)]

        Example: n_lru_nuked resets on a varnish restart
        >>> with tempfile.NamedTemporaryFile() as fd:
        ...     with nagiosplugin.Cookie(fd.name) as cookie:
        ...         CollectDeltaLRUNuked.collect({'MAIN.n_lru_nuked':dict(value=613)}, cookie)
        ...         CollectDeltaLRUNuked.collect({'MAIN.n_lru_nuked':dict(value=0)}, cookie)
        []
        [Metric(name='nuked_delta', value=0, uom='count', min=0, max=None, context='nuked_delta', contextobj=None, resource=None)]

        Example: n_lru_nuked decreases on a varnish restart followed closely by some nuke events
        >>> with tempfile.NamedTemporaryFile() as fd:
        ...     with nagiosplugin.Cookie(fd.name) as cookie:
        ...         CollectDeltaLRUNuked.collect({'MAIN.n_lru_nuked':dict(value=613)}, cookie)
        ...         CollectDeltaLRUNuked.collect({'MAIN.n_lru_nuked':dict(value=2)}, cookie)
        []
        [Metric(name='nuked_delta', value=2, uom='count', min=0, max=None, context='nuked_delta', contextobj=None, resource=None)]

        Cookie data should always match the previous nuked value
        >>> with tempfile.NamedTemporaryFile() as fd:
        ...     with nagiosplugin.Cookie(fd.name) as cookie:
        ...         assert cookie.get('MAIN.n_lru_nuked') == None
        ...         _ = CollectDeltaLRUNuked.collect({'MAIN.n_lru_nuked':dict(value=2)}, cookie)
        ...     with nagiosplugin.Cookie(fd.name) as cookie:
        ...         assert cookie.get('MAIN.n_lru_nuked') == 2
        ...         _ = CollectDeltaLRUNuked.collect({'MAIN.n_lru_nuked':dict(value=3)}, cookie)
        ...     with nagiosplugin.Cookie(fd.name) as cookie:
        ...         assert cookie.get('MAIN.n_lru_nuked') == 3
        ...         _ = CollectDeltaLRUNuked.collect({'MAIN.n_lru_nuked':dict(value=1)}, cookie)
        ...     with nagiosplugin.Cookie(fd.name) as cookie:
        ...         assert cookie.get('MAIN.n_lru_nuked') == 1
        '''
        # fetch current and previous values
        prev_nuked = state_cookie.get(cls._nuked)
        curr_nuked = json_data[cls._nuked]['value']
        # update state with whatever we got from varnishstat
        state_cookie[cls._nuked] = curr_nuked

        # no data was in the cookie file, so no delta is computed
        if prev_nuked is None:
            return []

        return [nagiosplugin.Metric(
            name=NUKED_NAME(),
            # nuked monotonically increases, otherwise there was a varnish restart
            value=curr_nuked if curr_nuked < prev_nuked else curr_nuked - prev_nuked,
            uom='count',
            min=0,
        )]


@nagiosplugin.guarded(verbose=0)
def main(args):
    if args.v:
        print args
    if args.test:
        print doctest.testmod()
        return 1
    # XXX: nagiosplugin.Check.main uses sys.exit internally
    nagiosplugin.Check(
        Varnishstats(args.name, args.statefile, [
            CollectSMAUsage(args.sma_name),
            CollectDeltaLRUNuked(),
        ]),
        nagiosplugin.ScalarContext(
            NUKED_NAME(),
            warning=args.nuked_warn,
            critical=args.nuked_crit
        ),
        nagiosplugin.ScalarContext(
            SMA_USAGE_NAME(),
            warning=args.sma_usage_warn,
            critical=args.sma_usage_crit
        ),
    ).main()


def parse(argv):
    ap = argparse.ArgumentParser(description=__doc__, formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    ap.add_argument('-n', '--name', type=str, metavar='NAME', default=default('varnish_name'), help='Name of the varnishd instance to get stats from')

    ap.add_argument('--sma-usage-warn', metavar='RANGE', type=nagiosplugin.Range, default=default('sma_usage_warn'), help='Allowed range for memory usage (a fraction on [0, 1]), otherwise warn')
    ap.add_argument('--sma-usage-crit', metavar='RANGE', type=nagiosplugin.Range, default=default('sma_usage_crit'), help='Allowed range for memory usage (a fraction on [0, 1]), otherwise crit')
    ap.add_argument('--sma_name', metavar='STR', help='The name of the varnish SMA for which to compute usage', default=default('sma_name'))

    ap.add_argument('--nuked-warn', metavar='RANGE', type=nagiosplugin.Range, help='Allowed range for delta of lru_nuked, otherwise warn')
    ap.add_argument('--nuked-crit', metavar='RANGE', type=nagiosplugin.Range, help='Allowed range for delta of lru_nuked, otherwise crit', default=default('nuked_crit'))

    ap.add_argument('--statefile', metavar='PATH', help='Where to store & retrieve this check\'s state', default=default('check_statefile'))
    ap.add_argument('-v', action='count', default=0, help='Verbosity level passed to nagiosplugin.Check.main; repeat for higher verbosity')
    ap.add_argument('--test', action='store_true', help='Run tests and quit immediately')
    return ap.parse_args(argv)


if __name__ == '__main__':
    sys.exit(main(parse(sys.argv[1:])))
