#!/usr/bin/env python

import argparse
import base64
import csv
import StringIO
import sys
import urllib2

def main():
    try:
        _main()
    except Exception as e:
        return_result('UNKNOWN', 'Unexpected exception: {}'.format(e))

def _main():
    parser = argparse.ArgumentParser(description='detect when haproxy connection pools are exhausted')
    parser.add_argument('--url', type=str, help='haproxy stats csv url')
    parser.add_argument('--user', type=str, help='haproxy stats auth user')
    parser.add_argument('--pw', type=str, help='haproxy stats auth password')
    parser.add_argument('-e', dest='entries', action='append', help='haproxy frontend/backends to alert on')
    parser.add_argument('--warn', type=float, help='percent of connections in a pool used to trigger warning alerts')
    parser.add_argument('--crit', type=float, help='percent of connections in a pool used to trigger critical alerts')
    args = parser.parse_args(sys.argv[1:])

    missing_fields = []
    if not args.url:
        missing_fields.append('--url')
    if not args.user:
        missing_fields.append('--user')
    if not args.pw:
        missing_fields.append('--pw')
    if not args.entries:
        missing_fields.append('-e')
    if not args.warn:
        missing_fields.append('--warn')
    if not args.crit:
        missing_fields.append('--crit')
    if missing_fields:
        return_result('UNKNOWN', 'Missing script arguments: {}'.format(' '.join(missing_fields)))

    invalid_fields = []
    if args.warn < 0 or args.warn > 1:
        invalid_fields.append('--warn')
    if args.crit < 0 or args.crit > 1:
        invalid_fields.append('--crit')
    if invalid_fields:
        return_result('UNKNOWN', 'Invalid script arguments: {}'.format(' '.join(invalid_fields)))

    entries = set(args.entries)
    criticals = []
    warns = []

    request = urllib2.Request(args.url)
    request.add_header('Authorization', http_basic_auth(args.user, args.pw))
    f = urllib2.urlopen(request)
    reader = csv.DictReader(f)
    for row in reader:
        if row['# pxname'] in entries:
            if int(row['scur']) > (args.crit * int(row['slim'])):
                criticals.append(entry_name(row))
            elif int(row['scur']) > (args.warn * int(row['slim'])):
                warns.append(entry_name(row))

    if criticals:
        return_result('CRITICAL', output_message(criticals, warns))
    if warns:
        return_result('WARNING', output_message(criticals, warns))
    return_result('OK', 'All haproxy connection pools are healthy')

def entry_name(row):
    return '{}[{}]'.format(row['# pxname'], row['svname'])

def output_message(criticals, warns):
    msg = ''
    if criticals:
        msg += 'CRIT:{} '.format(', '.join(criticals))
    if warns:
        msg += 'WARN:{}'.format(', '.join(warns))
    return msg

def return_result(status, message):
    ''' Prints a result and exits in the way Nagios expects
    '''
    # print the supplied message
    print('{0} - {1}'.format(status, message))

    # exit with the proper exit code
    states = {
        'OK': 0,
        'WARNING': 1,
        'CRITICAL': 2,
        'UNKNOWN': 3,
    }
    if status in states.keys():
        exit_code = states.get(status)
    else:
        exit_code = states.get('UNKNOWN')
    sys.exit(exit_code)

def http_basic_auth(user, pw):
    return "Basic {}".format(base64.encodestring("{}:{}".format(user, pw).rstrip("\n")))

if __name__ == '__main__':
    main()
