#!/usr/bin/env python
#
# Nagios check to see if a particular interface exists, is up, and has
# a particular IP address. Simple wrapper around "ip addr list"
#
# (Strictly speaking we deal in aliases, not interfaces. So lo,
# lo:1 and lo:8 are all different things)
#
# Example:
#  python ./interface_with_ip.py --interface lo:1 --ip 192.16.71.144
#

import argparse
import re
import subprocess
import sys

OK      = (0, 'OK')
WARN    = (1, 'WARN')
CRIT    = (2, 'CRIT')
UNKNOWN = (3, 'UNKNOWN')

def ipaddrlist():
    """
    Returns (debug text including stderr, [one multi-line string per interface]).

    ip addr list output looks like this (each N: interface: <flags>... block gets its own
    item in the returned list) -

        1: lo: <LOOPBACK,UP,LOWER_UP> mtu 16436 qdisc noqueue state UNKNOWN
            link/loopback 00:00:00:00:00:00 brd 00:00:00:00:00:00
            inet 127.0.0.1/8 scope host lo
        2: eth0: <BROADCAST,MULTICAST> mtu 1500 qdisc noop state DOWN qlen 1000
            link/ether 00:25:90:81:a2:cc brd ff:ff:ff:ff:ff:ff
    """
    (stdout,stderr) = subprocess.Popen("ip addr list", shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE).communicate()

    grouped_lines = [""]
    for l in stdout.split("\n"):
        if len(l) and not l[0].isspace():
            grouped_lines.append(l + "\n")
        else:
            grouped_lines[-1] += l + "\n"

    return stderr+stdout, grouped_lines

    
def get_status(interface, ip):
    extra_data = ""
    try:
        extra_data, interface_list = ipaddrlist()

        for interface_text in interface_list:
            lines = interface_text.split("\n")

            interface_up = bool(re.search("[<,]UP[,>]", lines[0]))

            RE_INET = re.compile('\sinet %s[^0-9]' % ip.replace('.', '\.'))

            has_alias = False
            for l in lines[1:]:
                inet_string = "inet " + ip
                inet_ofs = l.find(inet_string)
                if RE_INET.search(l) and l.endswith(" " + interface):
                    has_alias = True

            if has_alias:
                if interface_up:
                    return OK, "Interface %s is up and has address %s" % (interface,ip), extra_data
                else:
                    return CRIT, "Interface %s is DOWN and has address %s" % (interface,ip), extra_data

        return CRIT, "Could not locate IP %s on interface %s" % (ip,interface), extra_data

    except Exception, e:
        return UNKNOWN, "Exception while getting/parsing interface list: %s" % (e), extra_data


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('-n', '--interface', dest='interface', required=True, help='Interface (alias) to check eg bond0, eth3, lo:1')
    parser.add_argument('-p', '--ip', dest='ip', required=True, help='IP address to check for eg 192.168.44.12')
    args = parser.parse_args()

    status, detail, extra_data = get_status(args.interface.strip(), args.ip.strip())

    print "%s: %s" % (status[1], detail)
    print extra_data
    return status[0]

if __name__ == '__main__':
    sys.exit(main())

