#!/usr/bin/env python
import argparse
import memcache
import sys


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('--keyspace-usage', metavar='N%', type=int, default=90,
                        help='Emit WARNING if keyspace usage > N%')
    parser.add_argument('--uptime', metavar='N', type=int, default=600,
                        help='Emit WARNING if uptime is less than N seconds')
    return parser.parse_args()


def return_result(status, message):
    ''' Prints a result and exits in the way Nagios expects
    '''
    # print the supplied message
    print '{0} - {1}'.format(status, message)

    # exit with the proper exit code
    states = {
        'OK': 0,
        'WARNING': 1,
        'CRITICAL': 2,
        'UNKNOWN': 3,
    }
    if status in states.keys():
        exit_code = states.get(status)
    else:
        exit_code = states.get('UNKNOWN')
    sys.exit(exit_code)


def main(options):
    ''' Retrieves statistics from Memcached's telnet interface, and alerts on
    connection failure, hash expansion, or recent restarts.
    '''
    mc = memcache.Client(['127.0.0.1:11211'])
    stats = mc.get_stats()[0][1]  # get_stats() returns [(socket, stats)]

    # check for connection failure
    if stats['accepting_conns'] != '1':
        return_result('CRITICAL', 'Memcached is not accepting connections')

    # check for hash expansions in progress
    if stats['hash_is_expanding'] == '1':
        return_result('CRITICAL', 'Hash table is expanding')

    # check for upcoming hash expansions
    keys = int(stats['curr_items'])
    bucket_count = 2 ** int(stats['hash_power_level'])
    # buckets store multiple keys in a linked list
    # to do: determine which specific threshold triggers hash expansion
    keyspace_usage = 0.0
    threshold = options.keyspace_usage / 100.0

    if keyspace_usage >= threshold:
        formatter = ('Keyspace usage ({0}%) is over threshold {1}%. '
                     'Hash expansion will occur at 100%.')
        message = formatter.format(
            int(keyspace_usage * 100),
            options.keyspace_usage,
        )
        # temporarily disabled
        #return_result('WARNING', message)

    # check for recent restarts
    uptime = int(stats['uptime'])
    if uptime < options.uptime:
        message = 'Memcached uptime ({0}s) is under {1} seconds'.format(
            uptime,
            options.uptime,
        )
        return_result('WARNING', message)

    # exit normally
    return_result('OK', 'Memcached up for {0}s'.format(uptime))

if __name__ == '__main__':
    options = parse_args()
    main(options)
