#!/usr/bin/env python
""" This python script scans the openvpnas license folder and returns warning
critical if there is not at least one license within the thresholds.
https://jira.twitch.com/browse/SYS-10845 - David Newhall II - Aug 3, 2017
"""

from datetime import datetime
from glob import glob
from os.path import isfile, join
from pynagios import make_option, Plugin, Response, UNKNOWN, CRITICAL, WARNING, OK

LICENSE_FOLDER = '/usr/local/openvpn_as/etc/licenses/'


class OpenVPNLicense(Plugin):
    """ Class container for OpenVPN License Check, extends pynagios Plugin. """
    unknown = make_option(
        '-u', '--unknown', action='store_true',
        help='Return UNKNOWN instead of CRIT if no license files are found.')
    folder = make_option(
        '-f', '--folder', default=LICENSE_FOLDER,
        help='Folder in which to locate OpenVPN license files.')

    def check(self):
        """ This is the main block of code that does all the things. """
        license_files = self.get_license_files()
        if not license_files:
            msg = "No license files found in directory: {}".format(LICENSE_FOLDER)
            state = UNKNOWN if self.options.unknown else CRITICAL
            return Response(state, msg)

        expiry_dates = self.parse_each_license_file(license_files)
        expiry_days = self.compare_license_dates(expiry_dates)

        # Use the license file that returned the largest "days" before expiration.
        try:
            last_date = sorted(expiry_days)[-1:][0]
        except IndexError:
            state = UNKNOWN if self.options.unknown else CRITICAL
            msg = "Error occurred parsing license file(s)."
        else:
            state = OK
            msg = "Only {} days until OpenVPN License Expires!".format(last_date)
            if (hasattr(self.options, 'critical') and
                    self.options.critical is not None and
                    self.options.critical.in_range(last_date)):
                state = CRITICAL
            elif (hasattr(self.options, 'warning')
                  and self.options.warning is not None
                  and self.options.warning.in_range(last_date)):
                state = WARNING
            else:
                msg = "OpenVPN License OK! {} days until expiration.".format(last_date)

        msg = "{}\nFound {} total license files.".format(msg, len(license_files))
        return Response(state, msg)

    def get_license_files(self):
        """ Returns a list of license files in the license-file path. In addition
        to the currently running license, there is likely an old file or two. """
        try:
            files = [f for f in glob(join(self.options.folder, '*.lic')) if isfile(f)]
            if self.options.verbosity > 1:
                print "Found {} license files: {}".format(len(files), ', '.join(files))
            return files

        except OSError as err:
            if self.options.verbosity:
                print "Error locating license files: {}".format(err)
            return list()

    def parse_each_license_file(self, license_files):
        """ Parses all License Files and returns their expiration dates. """
        license_expiration_dates = list()
        for license_file in license_files:
            try:
                with open(license_file, 'r') as open_lic_file:

                    for line in open_lic_file:
                        if line.startswith('expiry_date='):
                            if self.options.verbosity > 1:
                                print "Found expiry_date in {} -> {}".format(license_file, line[:-1])
                            # Get the data after the = and strip off the newline.
                            license_expiration_dates.append(line.split('=')[1][:-1])
                            break

            except (OSError, IOError, IndexError) as err:
                if self.options.verbosity:
                    print "Error parsing license file {}: {}".format(license_file, err)
        return license_expiration_dates

    def compare_license_dates(self, license_expiration_dates):
        """ Compares the expiration date of each license to now and returns a list
        with the nuber of days before each license expires. """
        if self.options.verbosity > 1:
            print "Comparing dates to now: {}".format(', '.join(license_expiration_dates))

        try:
            return [(datetime.strptime(lic_date, '%Y%m%d') - datetime.now()).days
                    for lic_date in license_expiration_dates]
        except ValueError as err:
            if self.options.verbosity:
                print "There was an error parsing and compring dates: {}".format(err)
            return list()


if __name__ == "__main__":
    OpenVPNLicense().check().exit()
