#!/usr/bin/env python

import argparse, nagiosplugin, psycopg2, logging

class Lag(nagiosplugin.Resource):
    """Domain model: repmgr lag"""
    def __init__(self):
        super(Lag, self).__init__()

    def total_seconds(self, td):
        total = td.seconds + td.days * 24 * 3600
        total = float ( str(total) + "." + str(td.microseconds))
        return total

    def probe(self):
        logging.info('getting lag info from db')
        conn = psycopg2.connect("dbname=repmgr user=postgres")
        cur = conn.cursor()
        cur.execute("SELECT standby_node,time_lag FROM repmgr_sitedb.repl_status;")
        if cur.rowcount > 0:
            lags = cur.fetchall()
        else:
            lags = False
        cur.close()
        conn.close()
        for (slave, lag) in lags:
            sec = self.total_seconds(lag)
            logging.debug('node %02d lag: %s (%f seconds)' % (slave, lag, sec))
            yield nagiosplugin.Metric('slave %d' % slave, sec, min=0, context='lag')

class LagSummary(nagiosplugin.Summary):
    def __init__(self):
        super(LagSummary, self).__init__()
    
    def ok(self, results):
        max = 0
        for i in results:
            if i.metric.value > max:
                max = i.metric.value
                slave = i.metric.name
        logging.debug(repr(max))
        return "%s is max at %fs" % (slave, max)

@nagiosplugin.guarded
def main():
    argp = argparse.ArgumentParser(description = __doc__)
    argp.add_argument('-w', '--warning', metavar='RANGE', default='',
                        help='return warning if lag is outside RANGE')
    argp.add_argument('-c', '--critical', metavar='RANGE', default='',
                        help='return critical if lag is outside RANGE')
    argp.add_argument('-v', '--verbose', action='count', default=0,
                        help='increase output verbosity (use up to 3 times)')
    args = argp.parse_args()
    check = nagiosplugin.Check(
        Lag(),
        nagiosplugin.ScalarContext('lag', args.warning, args.critical),
        LagSummary()
    )
    check.main(verbose=args.verbose)
    logging.debug("Args: %s" % args)

if __name__ == '__main__':
    main()
