#!/usr/bin/env python

import argparse, logging, subprocess, os
import nagiosplugin

PG_HOME = os.path.expanduser('~postgres')
REPMGR_CONF = os.path.join(PG_HOME, 'repmgr/repmgr.conf')
CHECK_STATEFILE = '/var/tmp/repmgr_state'
logger = logging.getLogger('nagiosplugin')

class RepState(nagiosplugin.Resource):
    def __init__(self):
        super(RepState, self).__init__()
        with nagiosplugin.Cookie(CHECK_STATEFILE) as self.repconfig:
            if not self.repconfig.get('node_name', None):
                # read node name from config file on first run
                logger.info("reading repmgr config %s" % REPMGR_CONF)
                with open(REPMGR_CONF, 'r') as cf:
                    for line in cf.readlines():
                        if 'node_name' in line:
                            logger.debug("parsing line: %s" % line)
                            (key, val) = line.split('=')
                            self.repconfig[key] = val.strip()
                logger.debug("config: %s" % self.repconfig)

            # get current cluster role
            logger.debug('running repmgr cluster show')
            try:
                output = subprocess.check_output(['repmgr', '-f', REPMGR_CONF, 'cluster', 'show'])
            except OSError:
                raise nagiosplugin.CheckError('running repmgr failed!')

            for line in output.splitlines():
                info = line.split()
                if info[0] == '*': info.pop(0)
                if len(info) > 2 and info[2] == self.repconfig['node_name']:   
                    logger.debug("found nodename in line: %s" % line)
                    self.role = info[0]
                    logger.debug("i discovered my current role is %s" % self.role)
                    break

            if not self.repconfig.get('node_role', None):
                logger.info("initializng expected role as %s" % self.role)
                self.repconfig['node_role'] = self.role


    def probe(self):
        result = 1
        if self.repconfig['node_role'] == self.role: result = 0
        return nagiosplugin.Metric("role: %s (expected: %s)" % (self.role, self.repconfig['node_role']), result, context='role')



@nagiosplugin.guarded(verbose=0)
def main():
    argp = argparse.ArgumentParser(description = __doc__)
    argp.add_argument('-v', '--verbose', action='count', default=0)
    argp.add_argument('-w', '--warn')
    argp.add_argument('-c', '--crit')
    args = argp.parse_args()

    check = nagiosplugin.Check( RepState(),
        nagiosplugin.ScalarContext('role', fmt_metric='{name}', critical=nagiosplugin.Range('0:0')),
    )
    check.main(args.verbose)

if __name__ == '__main__':
    main()
