#!/usr/bin/env python
""" This python script runs puppet and returns an appropriate nagios code.
https://jira.twitch.com/browse/SYS-12991 - David Newhall II - Aug 17, 2018
"""

from subprocess import Popen, PIPE
from pynagios import make_option, Plugin, Response, UNKNOWN, CRITICAL, OK

DEFAULT_BRANCH = 'production'


class RunPuppet(Plugin):
    """ Class container for Puppet Return Code check, extends pynagios Plugin. """
    unknown = make_option(
        '-u', '--unknown', action='store_true',
        help='Return UNKNOWN instead of CRIT if puppet is disabled.')
    branch = make_option(
        '-b', '--branch', default=DEFAULT_BRANCH,
        help='This controls the --environment parameter passed to puppet agent.')
    sudo = make_option(
        '-s', '--sudo', action='store_true',
        help='Not running as root? Add this argument to run puppet with sudo.')

    def check(self):
        """ This is the main block of code that does all the things. """
        puppet_return_code, output = self.get_puppet_return_code()
        if puppet_return_code == -1:
            msg = "Puppet binary not found."
            state = CRITICAL
        elif puppet_return_code == 0:
            msg = "Puppet reported success and no changes! (0)"
            state = OK
        elif puppet_return_code == 1:
            msg = "Puppet is disabled, or already running, or you're not root. (1)\n"+output
            state = UNKNOWN if self.options.unknown else CRITICAL
        elif puppet_return_code == 2:
            msg = "Puppet reported successful changes applied! (2)"
            state = OK
        elif puppet_return_code == 4:
            msg = "Puppet Error applying changes. (4)\n"+output
            state = CRITICAL
        elif puppet_return_code == 6:
            msg = "Puppet Error applying changes; some changes were OK. (6)\n"+output
            state = CRITICAL
        else:
            msg = "Puppet Agent returned an unknown code. ({0})\n{1}".format(
                puppet_return_code, output)
            state = UNKNOWN
        return Response(state, msg)

    def get_puppet_return_code(self):
        """ Runs puppet and returns the exit code & output.
            Code -1 means the binary wasn't found.
        """

        cmd = ["puppet", "agent", "--test", "--color=false", "--environment="+self.options.branch]
        if self.options.sudo:
            cmd.insert(0, "sudo")
        try:
            process = Popen(cmd, stdout=PIPE, stderr=PIPE)
            output = "".join(process.communicate()).rstrip()
            svc_status = process.returncode
        except OSError as err:
            svc_status = -1
            output = "Caught Exception: {}".format(err)
        return svc_status, output


if __name__ == "__main__":
    RunPuppet().check().exit()
