#!/usr/bin/env python
import argparse
import requests
import json
import sys
import traceback

OK   = 0
WARN = 1
CRIT = 2
UNKN = 3

CHECKED_NODE_TYPES = ['hls']

parser = argparse.ArgumentParser()
parser.add_argument('-w', '--warn', dest='warn', type=float, default=0.1, help='warn level')
parser.add_argument('-c', '--crit', dest='crit', type=float, default=0.15, help='crit level')
parser.add_argument('-C', '--cluster', dest='cluster', help='cluster name', required=True)
parser.add_argument('-v', '--verbose', dest='verbose', default=False, action='store_true')
args = parser.parse_args()

def exit(status, message):
    status_string = {OK: "OK", CRIT: "CRIT", WARN: "WARN", UNKN: "UNKN"}[status]
    print "%s: %s" % (status_string, message)
    sys.exit(status)

def get_disabled_ratio(cluster):
    try:
        r = requests.get("http://usher.twitch.tv/node/list.json", timeout=10)
        r.raise_for_status()
        data = json.loads(r.text)
    except Exception as e:
        exit(UNKN, "Exception: '%s'\n%s" % (e, traceback.format_exc()))

    count = 0.0
    enabled_count = 0.0
    for node in data:
        if node['cluster_name'] != cluster or node['rep_type'] not in CHECKED_NODE_TYPES:
            continue

        count += 1

        if node['play_enabled'] == True:
            enabled_count += 1

        if args.verbose:
            print "%s -> %s" % (node['ip'], 'enabled' if node['play_enabled'] == True else 'disabled')

    if args.verbose:
        print "Overall: %s/%s enabled nodes" % (enabled_count, count)

    # If cluster has 0 nodes there are no disabled nodes
    if count < 1:
        return 0.0

    return 1 - enabled_count / count

def main():
    disabled_percent = get_disabled_ratio(args.cluster)
    exit_string = "Cluster %s has %s ratio of disabled nodes" % (args.cluster, disabled_percent)

    if disabled_percent > args.crit:
        exit(CRIT, exit_string)
    elif disabled_percent > args.warn:
        exit(WARN, exit_string)
    else:
        exit(OK, exit_string)

if __name__ == "__main__":
    main()
