#!/bin/bash

ESHOST="$1"
WARN="${2:-15}"
CRIT="${3:-9}"

# Critical level is set to 9 (by default) because *at* 10% (again, by default),
# the node will start attempting to reallocate its shards away.  It should only
# make it to 9% under extremely heavy writes that are fast enough that the
# shard relocation couldn't work or in the much more likely case when the
# cluster has nowhere to reallocate the shards.

if [ x = x"$ESHOST" ];then
    echo "UNKNOWN: Usage is $0 ESHOST [warn] [crit]"
    exit 3
fi

set -f -- `curl -s "http://${ESHOST}:9200/_nodes/stats?format=json" | python -c '
import json, sys
j = json.load(sys.stdin)

def calc_node_perc(x):
    tot=sum(y["total_in_bytes"] for y in x)
    fre=sum(y["available_in_bytes"] for y in x)
    return fre*100.0 / tot

r = tuple(sorted([(calc_node_perc(x["fs"]["data"]),x["host"]) for x in j["nodes"].values()])[0])
print "%02d %s"%r
'`

if [ $1 -le "$CRIT" ];then
    echo "CRITICAL: Node $2 is <= $CRIT% ($1) free disk space overall.  There may be no nodes with room for a shard transfer from this node.;$1;$2"
    exit 2
fi
if [ $1 -le "$WARN" ];then
    echo "WARNING: Node $2 is <= $WARN% ($1) free disk space overall.  This host may be unable to accept new shards soon.;$1;$2"
    exit 1
fi

echo "OK: All data nodes > $WARN% free disk space (minimum is $2 at $1% free);$1;$2"
