#!/usr/bin/env python

import argparse
import requests
import time
import sys
import hmac
import hashlib
import traceback

parser = argparse.ArgumentParser()
parser.add_argument('-n', '--node', required=True, type=str, help='node')
parser.add_argument('-k', '--stream-num', type=int, default=5, help='Number of streams to try downloading manifests for')
parser.add_argument('-c', '--critical', metavar='LEVEL', type=int, default=3, help="number of manifests which can't be d/l for CRIT")
parser.add_argument('-w', '--warn', metavar='LEVEL', type=int, default=2, help="number of manifests which can't be d/l for WARN")
parser.add_argument('-e', '--expected', metavar='LEVEL', type=int, default=0, help="expected number of und/lable manifests")

parser.add_argument('--key-source', type=str, default="http://usher.justin.tv/hls_transcode/manifest_keys.json", help="usher endpoint providing a list of actively used keys")

OPTIONS = parser.parse_args()
MAX_TOKEN=2**30

EDGE_SUFFIX='hls.ttvnw.net'
PR_SUFFIX='justin.tv'

FQDN_SUFFIX='.justin.tv'
if OPTIONS.node.endswith(FQDN_SUFFIX):
    OPTIONS.node = OPTIONS.node[:-len(FQDN_SUFFIX)]

def safe_json(resp):
    """ Wrapper for requests.json() for api compatibility between versions"""

    try:
        return resp.json()
    except TypeError:
        return resp.json

def is_edge():
    resp = requests.get("http://usher.justin.tv/node/show/{0}.json".format(OPTIONS.node))

    node = safe_json(resp)
    if not node or not node[0]:
        print 'OK: found 0 nodes named {0}'.format(OPTIONS.node)
        sys.exit(0)

    return node[0]['rep_type'] == 'hls'

def get_k_streams(k):
    resp = requests.get("http://usher.justin.tv/hls_transcode/list_n_streams.json?n={0}".format(k))
    resp.raise_for_status()

    streams = safe_json(resp)

    for stream in streams:
        if stream.get('formats'):
            stream['formats'] = stream['formats'].split(',')[0]

    return streams

def get_keys():
    resp = requests.get(OPTIONS.key_source)
    resp.raise_for_status()
    # convert unicode objects to strings
    return map(str, safe_json(resp))

def node_to_host(is_edge=True):
    node_parts = OPTIONS.node.split('.')
    return "{0}.{1}.{2}".format(node_parts[0], node_parts[1], EDGE_SUFFIX if is_edge else PR_SUFFIX)

def node_to_name():
    node_parts = OPTIONS.node.split('.')
    return "{0}-1.{1}.hls.justin.tv".format(node_parts[0], node_parts[1])

def sign_token(token, keys):
    key = keys[abs(hash(token)) % len(keys)]
    signature = hmac.new(key, token, hashlib.sha1).hexdigest()
    return "token={token}&sig={signature}".format(token=token, signature=signature)

def generate_token(stream_id, fmt, keys):
    token = "id={token_id},bid={stream_id},exp={expires_on},node={edge_host},nname=nagios,fmt={fmt}".format(
                stream_id=stream_id,
                expires_on=int(time.time()) + 100,
                edge_host=node_to_name(),
                fmt=fmt,
                token_id=abs(hash(stream_id)) % MAX_TOKEN)

    return sign_token(token, keys)

def generate_base_url(stream, port=80, is_edge=False):
    return "http://{0}:{4}{1}/{2}/index{3}.m3u8".format(node_to_host(is_edge),
                                                        stream['uri'],
                                                        stream['formats'],
                                                        '-live' if stream['schema'] == 'new' else '',
                                                        port)

def generate_edge_url(stream, keys):
    return "{0}?{1}".format(generate_base_url(stream, is_edge=True), generate_token(stream['uri'].split('_')[-2], stream['formats'], keys))

def print_failed_urls(urls):
    for url in urls:
        print 'Failed url: {0}'.format(url)

try:
    keys = get_keys()
    streams = get_k_streams(OPTIONS.stream_num)
    if len(streams) < OPTIONS.stream_num:
        print 'CRIT: Fewer than {0} hls_transcodes exist!'.format(OPTIONS.stream_num)
        sys.exit(2)

    failed_list = []
    for stream in streams:
        if stream.get('formats'):
            if is_edge():
                url = generate_edge_url(stream, keys)
            else:
                url = generate_base_url(stream, port=6081, is_edge=False)
            resp = requests.get(url, allow_redirects=False)

            if not resp.status_code == requests.codes.ok:
                failed_list.append("{0} ({1})".format(url, resp.status_code))
        else:
            failed_list.append("{0} has no formats.".format(stream['uri']))

    num_failed = len(failed_list)
    num_succeeded = OPTIONS.stream_num - num_failed

    if num_failed < OPTIONS.warn:
        print 'OK: {0}/{1} stream manifests returned OK'.format(num_succeeded, OPTIONS.stream_num)
        sys.exit(0)
    elif num_failed < OPTIONS.critical:
        print 'WARN: {0}/{1} stream manifests returned OK'.format(num_succeeded, OPTIONS.stream_num)
        print_failed_urls(failed_list)
        sys.exit(1)
    else:
        print 'CRIT: {0}/{1} stream manifests returned OK'.format(num_succeeded, OPTIONS.stream_num)
        print_failed_urls(failed_list)
        sys.exit(2)

except Exception, e:
    print 'UNKNOWN: Exception {0}'.format(e)
    print traceback.format_exc()
    sys.exit(3)

