#!/usr/bin/env python

import requests, argparse, re
import logging
import operator
import imp
import os
import sys

cwd = os.path.dirname(os.path.realpath(__file__))
rabbit = imp.load_source('rabbit', os.path.join(cwd, 'rabbit.py'))

logger = logging.getLogger()

ch = logging.StreamHandler()
logger.addHandler(ch)

status = "OK"
status_code = 0
longline_text = []
failed_queues = []
attribute = "messages"

class KeyObject(object):
    def __init__(self, fn, fmt_msg):
        self.fn = fn
        self.fmt_msg = fmt_msg

    def format(self, queue, attr):
        return self.fmt_msg.format(queue['name'], self.fn(queue, attr))

QUANTITY = KeyObject(
    fn = lambda q, a: q[a],
    fmt_msg = "'{0}'={1}"
)
RATE = KeyObject(
    fn = lambda q, a: q[a]['rate'],
    fmt_msg = "'{0}.rate'={1}"
)

parser= argparse.ArgumentParser(
                      description="Check's RabbitMQ",
                      epilog = "As an alternative to the commandline, params can be placed in a file, one per line, and specified on the commandline like '%(prog)s @params.conf'.",
                      fromfile_prefix_chars = '@'
)

parser.add_argument(
    "--catchall",
    help = "Catch all unconfigured queues",
    action = "store_true",
)

parser.add_argument(
    'heading',
    help = 'config heading to check (ignored if --catchall is specified)'
)

parser.add_argument(
    '--config', 
    default = 'rabbit_queues.conf',
    help = 'file to load for config of known groups/headers'
)

parser.add_argument(
    "--url",
    help = "Rabbit server API url",
    default = "http://localhost:15672/api/"
)

parser.add_argument(
    "-W", "--warn",
    help = "warning level, use only to override the val specified in the config file",
    type = float,
    default = None
)

parser.add_argument(
    "-C", "--critical",
    help = "critical level, use only to override the val specified in the config file",
    type = float,
    default = None
)

parser.add_argument(
    "--invert",
    dest = 'comparison_fn',
    help = "invert warning and critical measurements (WARN and CRIT become ceilings instead of floors)",
    action = "store_const",
    const = operator.lt,
    default = operator.gt
)

parser.add_argument(
    "--rate",
    help = "look at message rates rather than quantity",
    dest = 'key_obj',
    action = 'store_const',
    const = RATE,
    default = QUANTITY
)

parser.add_argument(
    "--no-filter",
    help = "Skip filtering in catchall mode. (ignored if --catchall isn't specified)",
    action = 'store_true'
)

parser.add_argument(
    "--user",
    default = "guest"
)

parser.add_argument(
    "--password",
    default = "guest"
)

parser.add_argument(
    '--vhost', 
    default='/',
    help = 'vhost for rabbit to connect to'
)

parser.add_argument(
    '--debug',
    action = 'store_true',
    help = 'show debug info'
)


g = parser.add_mutually_exclusive_group(required=True)

g.add_argument(
    "--message",
    help = "look at basic message stats",
    action = "store_true"
)

g.add_argument(
    "--unacknowledged",
    help = "look at message_unacknowledged stats",
    action = "store_true"
)

g.add_argument(
    "--ready",
    help = "look at message_ready stats",
    action = "store_true"
)

args = parser.parse_args()

warn, crit = rabbit.get_limits(args.heading,
                               args.config)

if args.debug:
    logger.setLevel(logging.DEBUG)

if args.warn == None:
    args.warn = warn
if args.critical == None:
    args.critical = crit
    
if args.catchall:
    if args.no_filter:
        api_res = rabbit.get_unfiltered_json(conf_file = args.config,
                                              url = args.url,
                                              vhost = args.vhost,
                                              auth = (args.user, args.password))
    else:
        api_res = rabbit.get_filtered_json(conf_file = args.config,
                                              url = args.url,
                                              vhost = args.vhost,
                                              auth = (args.user, args.password))
else:
    api_res = rabbit.get_heading_json(args.heading, 
                                      conf_file = args.config,
                                      url = args.url,
                                      vhost = args.vhost,
                                      auth = (args.user, args.password))

# TODO: make rabbit return something so we have this info
# if r.status_code != 200:
#     print "RABBITMQ UNKNOWN - API Status Code: %s" % r.status_code
#     exit(3)

if args.unacknowledged:
    attribute += "_unacknowledged"
elif args.ready:
    attribute += "_ready"
    
if args.key_obj == RATE:
    attribute += "_details"

if len(api_res) == 0:
    status_code = 3
    status = 'UNKNOWN'
    failed_queues.append(args.heading + ' queue does not exist!')

for queue in api_res:
    format_str = "{0[name]} ({0["+attribute+"]:.0f}msgs)"
    longline_text.append(args.key_obj.format(queue, attribute))
    if args.comparison_fn(args.key_obj.fn(queue, attribute), args.warn):
        if status_code < 1:
            status_code = 1
            status = "WARN"
        failed_queues.append(format_str.format(queue))
    if args.comparison_fn(args.key_obj.fn(queue, attribute), args.critical):
        status_code = 2
        status = "CRITICAL"
        failed_queues.append(format_str.format(queue))

sys.stdout.write("%s: " % (status))
failed_queues = set(failed_queues)
if status != 'OK':
    print ", ".join(failed_queues)
else:
    sys.stdout.write('All good!')
sys.stdout.write(' | ')
print ' '.join(longline_text)
exit(status_code)
