import argparse
import requests
import json
import re
import sys
import urllib
from ConfigParser import SafeConfigParser
import logging

try:  # Python 2.7+
    from logging import NullHandler
except ImportError:
    class NullHandler(logging.Handler):
        def emit(self, record):
            pass

logger = logging.getLogger(__name__)

logger.addHandler(NullHandler())

parser = argparse.ArgumentParser()
parser.add_argument('heading'),
parser.add_argument('--config', default='rabbit_queues.conf')
parser.add_argument('--vhost', default='/')
parser.add_argument('--user', default='guest')
parser.add_argument('--passwd', default='guest')
parser.add_argument('--url', default='http://rabbit1.sfo01.justin.tv:15672/api/')

class ConfigParseError(Exception):
    pass

class RabbitApi(object):
    def __init__(self, vhost='/', auth=None, url=None):
        self.vhost = vhost
        self.auth = auth
        self.base_url = url or 'http://rabbit1.sfo01.justin.tv:15672/api/'

    def all_queues(self, vhost=None):
        v = urllib.quote_plus(vhost or self.vhost)
        r = requests.get('{url}queues/{vhost}'.format(vhost=v, url=self.base_url), auth=self.auth)
        return r

    def queue_by_name(self, name, vhost=None):
        v = urllib.quote_plus(vhost or self.vhost)
        r = requests.get('{url}queues/{vhost}/{name}'.format(vhost=v, url=self.base_url, name=name), 
                         auth=self.auth)
        return r

def get_queues(api):
    r = api.all_queues()
    r.raise_for_status()
    l = json.loads(r.content)
    return l

def get_queue(api, q):
    r = api.all_queues()
    r.raise_for_status()
    l = [x for x in json.loads(r.content) if x.get('name') == q]
    return l

def filter_queues(name, queues, cp):
    for conf in cp.sections():
        if cp.has_option(conf, 'prune'):
            if conf == name:
                queues = [x for x in queues if re.match(cp.get(conf, 'prune'), x)]
            else:
                if cp.get(conf, 'super') == name:
                    queues = [x for x in queues if not re.match(cp.get(conf, 'prune'), x)]
        if cp.has_option(conf, 'name'):
            queues = [x for x in queues if x != cp.get(conf, 'name')]
    return queues

class InvalidHeading(Exception):
    pass


def _filter_known_queue_names(names, cp):
    all_names = set(names)
    for conf in cp.sections():
        if cp.has_option(conf, 'prune'):
            names_to_remove = set([x for x in names if re.match(cp.get(conf, 'prune'), x)])
            all_names = all_names - names_to_remove
        if cp.has_option(conf, 'name'):
            all_names = all_names - set(cp.get(conf, 'name'))
    return all_names
    
def filter_known_queues(cp, api):
    """returns a list of structs for all queues not covered by the conf file"""
    queues = get_queues(api)
    queue_names = [x.get('name') for x in queues]
    res = set(_filter_known_queue_names(queue_names, cp))
    return [q for q in queues if q.get('name') in res]    
    
def _get_queues_by_heading(cp, api, name):
    if not cp.has_section(name):
        raise InvalidHeading
    logger.debug('header/values: {0} / {1}'.format(name, cp.items(name)))
    if cp.has_option(name, 'prune'):
        queues = get_queues(api)
        queue_names = [x.get('name') for x in queues]
        res = set(filter_queues(name, queue_names, cp))
        return [q for q in queues if q.get('name') in res]
    else:
        return get_queue(api, cp.get(name, 'name'))

def _load_conf(conf_file):
    cp = SafeConfigParser()
    try:
        cp.read([conf_file])
    except Exception:
        raise ConfigParseError
    return cp

def get_limits(heading, conf_file):
    cp = _load_conf(conf_file)
    string = cp.get(heading, 'limit')
    try:
        return tuple(map(int, string.split(':', 1)))
    except Exception:
        raise ConfigParseError

def get_heading_json(heading, conf_file=None, url=None, auth=None, vhost='/'):
    cp = _load_conf(conf_file)
    api = RabbitApi(auth=auth, url=url, vhost=vhost)
    return _get_queues_by_heading(cp, api, heading)

def get_filtered_json(conf_file=None, url=None, auth=None, vhost='/'):
    cp = _load_conf(conf_file)
    api = RabbitApi(auth=auth, url=url, vhost=vhost)
    return filter_known_queues(cp, api)

def get_unfiltered_json(conf_file=None, url=None, auth=None, vhost='/'):
    cp = _load_conf(conf_file)
    api = RabbitApi(auth=auth, url=url, vhost=vhost)
    return get_queues(api)


if __name__ == '__main__':
    args = parser.parse_args()
    auth = (args.user, args.passwd)
    print get_json(args.heading, 
                   conf_file=args.config,
                   url=args.url,
                   vhost=args.vhost,
                   auth=auth)