#!/usr/bin/env python
import json
import sys
import argparse
import smtplib
import logging
from logging.handlers import SysLogHandler
from twilio.rest.resources import Connection
from twilio.rest import TwilioRestClient
from email.mime.text import MIMEText
from time import sleep
from os import path

def configure_logger():
    l = logging.getLogger("syslog")
    formatter = logging.Formatter("smsmail " + "[%(process)s]: %(levelname)s - %(message)s")
    s = SysLogHandler('/dev/log', facility=SysLogHandler.LOG_LOCAL2)
    s.setFormatter(formatter)
    l.addHandler(s)

    return l

def main():
    global log
    send_via_email = False
    log = configure_logger()

    parser = argparse.ArgumentParser()
    parser.add_argument('-f', '--config', required=True, help='Config File')
    parser.add_argument('-t', '--to', metavar="NUMBER", required=True, help='Number to send sms to')
    parser.add_argument('-m', '--message', metavar="MESSAGE", help='Message to send')
    parser.add_argument('-e', '--email', metavar="EMAIL", help='Backup email in case of failure')
    parser.add_argument('--from', dest="from_address", metavar="EMAIL", required=True, help='Email to send from')
    parser.add_argument('-p', dest='proxy', metavar='PROXY', default=None, help='proxy server to use')
    args = parser.parse_args()

    if args.proxy:
        Connection.set_proxy_info(
                proxy_host=args.proxy,
                proxy_port=9797,
                )

    if not args.message:
        args.message = "".join(sys.stdin.readlines())

    send_sms_success = send_sms(args.config, args.to, args.message)

    if not send_sms_success:
        send_email(args.from_address, args.email, args.message)

def send_sms(config_path, to, message):
    global log
    log.warn("Sending sms to %s: %s" % (to, message))
    success = True

    if path.exists(config_path):
        try:
            with open(config_path) as f:
                raw_config=f.read()
                config = json.loads(raw_config)
        except:
            log.error("Can't parse config.")
            success = False
    else:
        log.error("Can't open config %s" % config_path)
        success = False

    if success == True:
        try:
            client = TwilioRestClient(config["account_number"], config["auth_token"])
            message_result = client.sms.messages.create(to=to, from_=config["from_number"], body=message[0:139])
            if message_result.status == "failed":
                success = False
        except Exception, e:
            log.error(e)
            success = False

    return success

def send_email(from_email, to_email, message):
    global log
    log.warn("Sending email to %s: %s" % (to_email, message))
    msg = MIMEText(message)
    msg['Subject'] = message.split('\n')[0]
    msg['From'] = from_email
    msg['To'] = to_email

    s = smtplib.SMTP('localhost')
    s.sendmail(from_email, [to_email], msg.as_string())
    s.quit()

if __name__ == "__main__":
    main()
