#!/usr/bin/env python3

import socket
import json
import traceback
import time

LIVESTATUS_HOST = "127.0.0.1"
LIVESTATUS_PORT = 6557
SOURCE = "livestatus"
RECV_SIZE = 4096
MAX_MTU = 1400
STATSD_HOST = "127.0.0.1"
STATSD_PORT = 8125
METRICS_PERIOD = 60

# all returned values: https://www.naemon.org/documentation/usersguide/livestatus.html#status
EXCLUDE = [
    "livestatus_version",
    "nagios_pid",
    "program_version"
]

def push_metrics():
    address = (LIVESTATUS_HOST, LIVESTATUS_PORT)
    sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    sock.connect(address)
    sock.sendall("GET status\nOutputFormat: json\n".encode('ascii'))
    sock.shutdown(socket.SHUT_WR)
    chunks = []
    while True:
        chunk = sock.recv(RECV_SIZE).decode('ascii')
        if len(chunk) == 0:
            break
        chunks.append(chunk)
    sock.close()
    reply = json.loads("".join(chunks))
    out = [[]]
    buf_len = 0
    for i, name in enumerate(reply[0]):
        if name in EXCLUDE:
            continue
        try:
            value = float(reply[1][i])
        except ValueError as ve:
            continue
        # please see https://github.com/influxdata/telegraf/pull/3183/files for more details about the cloudwatch limits
        if value > 0 and value < 8.515920e-109:
            value = 0
        metric = "{}:{}|g|#source:{}".format(name, value, SOURCE)
        metric_len = len(metric) + 1 # + 1 is because of newline
        if buf_len + metric_len > MAX_MTU:
            out.append([metric])
            buf_len = metric_len
        else:
            buf_len += metric_len
            out[-1].append(metric)
    
    sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    for metrics in out:
        data = "\n".join(metrics).encode('ascii')
        sock.sendto(data, (STATSD_HOST, STATSD_PORT))

def main():
    while True:
        next_run = time.time() + METRICS_PERIOD
        try:
            push_metrics()
        except Exception as e:
            print(traceback.format_exc())
        delay = next_run - time.time()
        if delay > 0:
            time.sleep(delay)

if __name__ == '__main__':
    try:
        main()
    except Exception as e:
        print(traceback.format_exc())
