#!/usr/bin/env python3

import subprocess
import socket
import traceback
import time

SOURCE = "nagiostats"
MAX_MTU = 1400
STATSD_HOST = "127.0.0.1"
STATSD_PORT = 8125
METRICS_PERIOD = 60

metrics_names = [
    "PROGRUNTIMETT",
    "STATUSFILEAGETT",
    "NAGIOSPID",
    "NUMSERVICES",
    "NUMHOSTS",
    "NUMSVCOK",
    "NUMSVCWARN",
    "NUMSVCUNKN",
    "NUMSVCCRIT",
    "NUMSVCPROB",
    "NUMSVCCHECKED",
    "NUMSVCSCHEDULED",
    "NUMSVCFLAPPING",
    "NUMSVCDOWNTIME",
    "NUMHSTUP",
    "NUMHSTDOWN",
    "NUMHSTUNR",
    "NUMHSTPROB",
    "NUMHSTCHECKED",
    "NUMHSTSCHEDULED",
    "NUMHSTFLAPPING",
    "NUMHSTDOWNTIME",
    "NUMHSTACTCHK1M",
    "NUMHSTPSVCHK1M",
    "NUMSVCACTCHK1M",
    "NUMSVCPSVCHK1M",
    "AVGACTSVCLAT",
    "AVGACTSVCEXT",
    "AVGACTSVCPSC",
    "AVGPSVSVCLAT",
    "AVGPSVSVCPSC",
    "AVGSVCPSC",
    "AVGACTHSTLAT",
    "AVGACTHSTEXT",
    "AVGACTHSTPSC",
    "AVGPSVHSTLAT",
    "AVGPSVHSTPSC",
    "AVGHSTPSC",
    "NUMACTHSTCHECKS1M",
    "NUMOACTHSTCHECKS1M",
    "NUMCACHEDHSTCHECKS1M",
    "NUMSACTHSTCHECKS1M",
    "NUMPARHSTCHECKS1M",
    "NUMSERHSTCHECKS1M",
    "NUMPSVHSTCHECKS1M",
    "NUMACTSVCCHECKS1M",
    "NUMOACTSVCCHECKS1M",
    "NUMCACHEDSVCCHECKS1M",
    "NUMSACTSVCCHECKS1M",
    "NUMPSVSVCCHECKS1M",
    "NUMEXTCMDS1M"
]
    
    
def collect_nofif_disabled_stats():
    """
    Pull current number of notifications that are disabled (state 0) on the video-nagios host. State 1 is indication of "enabled"
    """
    get_hst_notif_disabled_command = 'printf "GET hosts\nColumns: host_name description notifications_enabled\nFilter: notifications_enabled = 0\nOutputFormat: json\n" | sudo unixcat /var/nagios/rw/live | wc -l'
    get_svc_notif_disabled_command = 'printf "GET services\nColumns: host_name description notifications_enabled\nFilter: notifications_enabled = 0\nOutputFormat: json\n" | sudo unixcat /var/nagios/rw/live | wc -l'
    get_hst_notif_disabled_output = subprocess.run([get_hst_notif_disabled_command], shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    get_svc_notif_disabled_output = subprocess.run([get_svc_notif_disabled_command], shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    #The counter will return as 1 for an empty list so we subtract 1 to normalize the count
    hst_metric = (float(get_hst_notif_disabled_output.stdout) - 1)
    svc_metric = (float(get_svc_notif_disabled_output.stdout) - 1)
    #Format the metric to match the others
    hst_notif_disable_metric = "{}:{}|g|#source:{}".format('NUMHSTNOTIFDISABLED', hst_metric, SOURCE)
    svc_notif_disable_metric = "{}:{}|g|#source:{}".format('NUMSVCNOTIFDISABLED', svc_metric, SOURCE)
    return hst_notif_disable_metric, svc_notif_disable_metric
    
    
def push_metrics():
    stdout, stderr = subprocess.Popen(
        ["sudo", "/usr/bin/nagiostats", "-m", "-d", ",".join(metrics_names)],
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE
    ).communicate()
    values = stdout.decode('ascii').split('\n')
    out = [[]]
    buf_len = 0
    for i, name in enumerate(metrics_names):
        try:
            value = float(values[i])
        except ValueError as ve:
            continue
        # please see https://github.com/influxdata/telegraf/pull/3183/files for more details about the cloudwatch limits
        if value > 0 and value < 8.515920e-109:
            value = 0
        metric = "{}:{}|g|#source:{}".format(name, value, SOURCE)
        metric_len = len(metric) + 1 # + 1 is because of newline
        # If the buffer iterator and additional data (metric_len) exceed the MTU, create a new list
        if buf_len + metric_len > MAX_MTU:
            out.append([metric])
            buf_len = metric_len
        else:
            buf_len += metric_len
            out[-1].append(metric)
    
    # Call the function to get the number of notifcations disabled on the video-nagios host
    hst_notif_disable_metric, svc_notif_disable_metric = collect_nofif_disabled_stats()
    # Add the NUMNOTIFDISABLED metric to the grouping to be pushed to CloudWatch Metrics within its own buffer
    out.append([hst_notif_disable_metric])
    out.append([svc_notif_disable_metric])
    
    sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    for metrics in out:
        data = "\n".join(metrics).encode('ascii')
        sock.sendto(data, (STATSD_HOST, STATSD_PORT))

        
def main():
    while True:
        next_run = time.time() + METRICS_PERIOD
        try:
            push_metrics()
        except Exception as e:
            print(traceback.format_exc())
        delay = next_run - time.time()
        if delay > 0:
            time.sleep(delay)


if __name__ == '__main__':
    try:
        main()
    except Exception as e:
        print(traceback.format_exc())
