#!/usr/bin/env python

"""
This script runs on the video-nagios server and unsilences checks that meet the following requirements:
- Check is silenced	
- Check is NOT downtimed
- Check is in a good state
- Check hasn't had a state change in the past 72 hours 
- Check doesn't match the `ignored.yaml` file
"""

import json, yaml, os, socket, datetime, re, unicodedata, time, threading, logging, logging.handlers

# for local site only: file path to socket
address = "/var/nagios/rw/live"
# path to command file
commandfile = "/var/nagios/rw/nagios.cmd"

# setup logger
#logging.basicConfig(filename='/var/log/jtv/auto_unsilence.log', filemode='a', level=logging.INFO, format='%(levelname)s: %(message)s')
logger = logging.getLogger()
logger.setLevel(logging.INFO)

syslog = logging.handlers.SysLogHandler(address=("localhost", 514), facility='local3')
logger.addHandler(syslog)

formatter = logging.Formatter('%(module)s: %(message)s')
syslog.setFormatter(formatter)

# notify of startup
logger.info("\nStarting auto_unsilence.py @ %s" %(datetime.datetime.utcnow()))

# load ignored pop values
stream = file(os.path.dirname(os.path.realpath(__file__)) + '/ignored.yaml', 'r')
ignored = yaml.load(stream)

# setup empty list for loading nagios commands
commands = []

def livestatus_query(query):
    # connect to Livestatus API via UNIX Socket
    sock = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
    sock.connect(address)
    sock.sendall(query)
    sock.shutdown(socket.SHUT_WR)
    
    # receive the reply as a JSON string
    chunks = []
    while len(chunks) == 0 or chunks[-1] != "":
        chunks.append(sock.recv(4096))
    sock.close()
    return json.loads("".join(chunks))

# connect to Livestatus for services
silenced_services = livestatus_query("GET services\nColumns: host_name description service_notifications_enabled service_scheduled_downtime_depth state service_last_state_change\nFilter: state = 0\nFilter: service_notifications_enabled = 0\nAnd: 2\nOutputFormat: json\n")

# connect to Livestatus for hosts
silenced_hosts = livestatus_query("GET hosts\nColumns: host_name host_notifications_enabled host_scheduled_downtime_depth state host_last_state_change\nFilter: state = 0\nFilter: host_notifications_enabled = 0\nAnd: 2\nOutputFormat: json\n")

# parse the reply for services
for service in silenced_services:
  hostname = service[0]
  service_name = service[1]
  notification_state = service[2]
  downtime_state = service[3]
  state = service[4]
  last_state_change = datetime.datetime.utcfromtimestamp(service[5])
  now = datetime.datetime.utcnow()
  
  if downtime_state == 0:
    if len(hostname.split('.')) > 1:
      if hostname.split(".")[1] in ignored['pops']:
        logger.info("%s is a %s host, skipping" %(hostname, hostname.split('.')[1]))
        continue
      for ignored_hostname in ignored['hostnames']:
        pattern = re.compile(ignored_hostname)
        if pattern.search(hostname):
          logger.info("%s matches the ignored hostname filter (%s). Skipping the Nagios notification auto-enable" %(hostname, ignored['hostnames']))
          continue 
    for ignored_service in ignored['services']:
      pattern = re.compile(ignored_service)
      if pattern.search(service_name):
        logger.info("%s is a %s check, skipping" %(service_name, ignored_service))
        continue
    
    if now > last_state_change:
      if (now - last_state_change).days > 3:
        logger.info("%s - %s should be enabled" %(hostname, service_name))
        command = "[timefield] ENABLE_SVC_NOTIFICATIONS;%s;%s\n" %(hostname, service_name)
        logger.debug(command)
        commands.append(command)
      else:
        logger.info("we should wait on %s - %s" %(hostname, service_name))
  else:
    logger.info("%s - %s is downtimed, skipping" %(hostname, service_name))

# parse the reply for hosts
for host in silenced_hosts:
  hostname = host[0]
  notification_state = host[1]
  downtime_state = host[2]
  state = host[3]
  last_state_change = datetime.datetime.utcfromtimestamp(host[4])
  now = datetime.datetime.utcnow()
  if downtime_state == 0:
    if len(hostname.split('.')) > 1:
      if hostname.split(".")[1] in ignored['pops']:
        logger.info("%s is a %s host, skipping" %(hostname, hostname.split(".")[1]))
        continue
      for ignored_hostname in ignored['hostnames']:
        pattern = re.compile(ignored_hostname)
        if pattern.search(hostname):
          logger.info("%s matches the ignored hostname filter (%s). Skipping the Nagios notification auto-enable" %(hostname, ignored['hostnames']))
          continue
    
    if now > last_state_change:
      if (now - last_state_change).days > 3:
        logger.info("%s should be enabled" %(hostname))
        command = "[timefield] ENABLE_HOST_NOTIFICATIONS;%s\n" %(hostname)
        logger.debug(command)
        commands.append(command)
      else:
        logger.info("we should wait on %s" %(hostname))
  else:
    logger.info("%s is downtimed, skipping" %(hostname))

logger.debug("%i commands to process" %(len(commands)))

with open(commandfile, "a") as cmdfile:
  for command in commands:
    now = datetime.datetime.utcnow().strftime("%s")
    timed_command = command.replace("timefield",now)
    cmdfile.write(timed_command)
    logger.debug(timed_command)
    time.sleep(0.5)
cmdfile.close()

# notify of ending
logger.info("\nauto_unsilence.py completed @ %s" %(datetime.datetime.utcnow()))

exit(0)
