#!/usr/bin/env python
from time import sleep
import logging
from logging.handlers import SysLogHandler
import os
import socket
import re
import hashlib
from datetime import timedelta
from datetime import datetime
import yaml
import jira
from requests import exceptions
from collections import defaultdict

NAGIOS_CMD_LOCATION = '/var/nagios/rw/nagios.cmd'
NAGIOS_LIVE_SOCK = '/var/nagios/rw/live'
#This isn't needed? It's never referenced
NAGIOS_LOG_LOCATION = '/var/nagios/cache/nagios.log'

#Set the Proxy environment
proxy = 'http://proxy.internal.justin.tv:9797/'

os.environ['http_proxy'] = proxy
os.environ['HTTP_PROXY'] = proxy
os.environ['https_proxy'] = proxy
os.environ['HTTPS_PROXY'] = proxy

LOGGER = logging.getLogger('syslog')
formatter = logging.Formatter("jira_downtime [%(process)s]: %(message)s")
handler = SysLogHandler('/dev/log', facility=logging.handlers.SysLogHandler.LOG_LOCAL3)
handler.setFormatter(formatter)
LOGGER.addHandler(handler)
LOGGER.setLevel(logging.INFO)
jira.exceptions.JIRAError.log_to_tempfile = False

class Downtime(object):
    def __init__(self, host_name, end_time, comment, service, downtime_id):
        ticket_regex = re.compile('(?i)(VIDOPS-[0-9]+|VIDTOOL-[0-9]+|DC-[0-9]+|NET-[0-9]+|GSOC-[0-9]+|VIDCS-[0-9]+)')
        self.host_name = host_name
        self.end_time = end_time
        self.comment = comment
        self.service = service
        self.id_num = downtime_id
        self.ticket_num = ticket_regex.findall(comment)
        #self.user = user


    def _find_ticket(self, jira_connection):
        """Connect to JIRA and look for tickets with the same key found in comment
        We only need a boolean to satisfy if there IS a ticket open """
        for jira_ticket in self.ticket_num:
            try:
                search_string = 'id = "{}" AND status != "Closed"'.format(jira_ticket.upper())
                ticket_search = jira_connection.search_issues(search_string, maxResults=1)
                LOGGER.info("Searching JIRA for: {}".format(search_string))
                if ticket_search:
                    LOGGER.info("JIRA ticket found: {}".format(str(ticket_search)))
                    open_ticket_bool = True
                    #As soon as we find ONE ticket open, we can skip looking more.
                    return open_ticket_bool
                else:
                    LOGGER.info("JIRA ticket not found for {}".format(search_string))
                    open_ticket_bool = False
            except Exception as e:
                LOGGER.error("JIRA TICKET, {}, SEARCH FAILURE: {}".format(jira_ticket, str(e)))
                open_ticket_bool = False
        return open_ticket_bool

    def check_downtime(self, jira_connection):
        """Check downtime endtime, if less than three days, inject new downtime"""
        LOGGER.info("Checking if DT end is within defined threshold for {} / {} (nagios id: {} and end-date: {})".format(     \
            self.host_name, self.ticket_num, self.id_num, self.end_time))

        current_time = datetime.now()
        #Check iof the downtime is set to expire in less than 24 hours
        if self.end_time < current_time + timedelta(hours=24):
            LOGGER.info("Downtime end for {} {} ({}) is within threshold for expiration. Setting new end time in script so the downtime can be updated".format(self.host_name, self.service, self.ticket_num))
            #Set new end time to expire in 25 hours if DT is found to be within threshold
            self.end_time = current_time + timedelta(hours=25)

            if self.ticket_num and self._find_ticket(jira_connection):             
                self.new_downtime()
        return


    def new_downtime(self):
        """Create new downtime string"""
        current_time = datetime.now()

        #nagios likes epoch time, so let's convert-
        epoch_end_time = self.end_time.strftime('%s')
        epoch_current_time = current_time.strftime('%s')

        #if there is a string for "service" then downtime as a service, else host.
        if self.service != "host":
            downtime_string = "COMMAND [{0}] SCHEDULE_SVC_DOWNTIME;{1};{2};{3};{4};1;0;7200;auto-nagios;{5}\n".format( \
                epoch_current_time, self.host_name, self.service, epoch_current_time, epoch_end_time, self.comment)
        else:
            downtime_string = "COMMAND [{0}] SCHEDULE_HOST_DOWNTIME;{1};{2};{3};1;0;7200;auto-nagios;{4}\n".format(    \
                epoch_current_time, self.host_name, epoch_current_time, epoch_end_time, self.comment)

        LOGGER.info("Creating downtime for {} ({}) against {} with endtime as {}".format( \
            self.host_name, self.service, self.ticket_num, self.end_time))

        connect_to_livestatus(downtime_string)
        return

    def remove_downtime(self):
        """Craft string to remove downtime"""
        current_time = datetime.now().strftime('%s')
        remove_string = "COMMAND [{}] DEL_HOST_DOWNTIME;{}\n".format(current_time, self.id_num)
        #TODO
        #We should clean up SERVICE downtimes as well as it does not currently function for that
        LOGGER.info("Deleting extra downtime for {} {} {} with id: {} and endtime: {}".format( \
            self.host_name, self.service, self.ticket_num, self.id_num, self.end_time))
        connect_to_livestatus(remove_string)
        return

############# GO THROUGH DOWNTIMES ################

def sort_downtimes(list_of_downtimes):
    """Take unsort list of downtimes from Nagios, build dictionary
    make it bucket downtimes so we can clean-up/merge every excess downtime"""

    #This function actually creates a hash for each downtime and then puts that data into a dictionary {key, value} where the key is the hash and the value is a Downtime Object. 
    #This value, sorted_dict_of_downtimes[1], is later referenced as the "downtime_list" which contains values from the Class at the beginning of this script

    sorted_dict_of_downtimes = defaultdict(list)

    for downtime_entry in list_of_downtimes:
        host_name = downtime_entry[0]
        end_time = datetime.fromtimestamp(int(downtime_entry[1]))
        comment = downtime_entry[2]
        service = downtime_entry[3]
        downtime_id = downtime_entry[4]
        
        if not service:
            service = "host"

        current_downtime = Downtime(host_name, end_time, comment, service, downtime_id)

        host_service_string = "{}{}".format(current_downtime.host_name, current_downtime.service)
        downtime_hash = hashlib.md5(host_service_string).hexdigest()

        #insert via the hash as key
        sorted_dict_of_downtimes[downtime_hash].append(current_downtime)
    return sorted_dict_of_downtimes

def iterate_through_downtimes(sorted_dict_of_downtimes):
    """ Go through dict of downtimes, remove excess, make tickets
    E X T E N D   D O W N T I M E S """
    current_date = datetime.now()
    jira_connection = connect_to_jira()
    for downtime_list in sorted_dict_of_downtimes:
        if len(sorted_dict_of_downtimes[downtime_list]) > 1:
            new_downtime = merge_downtimes(sorted_dict_of_downtimes[downtime_list])
            new_downtime.check_downtime(jira_connection)
        else:
            sorted_dict_of_downtimes[downtime_list][0].check_downtime(jira_connection)
    return

def merge_downtimes(downtime_list):
    """Find the endtime furthest out, and all ticket numbers
    make new downtime, delete the old ones"""

    new_end_time = datetime.now()
    comment_string = ""
    host_name = downtime_list[0].host_name
    service_name = downtime_list[0].service

    for downtime_entry in downtime_list:
        LOGGER.info("Multiple downtimes found for {} {} {}. Merging object (downtime end: {} downtime_id: {})".format(host_name,\
                     service_name, downtime_entry.ticket_num, downtime_entry.end_time, downtime_entry.id_num))
        if downtime_entry.end_time > new_end_time:
            new_end_time = downtime_entry.end_time
        #I think this is needed to account for if there are multiple tickets in the ticket_num from the comment?
        #Take the first ticket in the regEx'd comment and save it back as the comment/ticket to use for extension?
        if downtime_entry.ticket_num:
            for ticket in downtime_entry.ticket_num:
                if ticket not in comment_string:
                    comment_string = "{} {}".format(comment_string, ticket)

    #This creates a new DT based on the fact that there are mutilple that need merging
    #The script has not yet checked JIRA
    LOGGER.info("Calling the new_downtime funciton to create a new downtime for {} {} with end_date as {} and comment as {}".format(host_name, \
        service_name, new_end_time, comment_string))
    merged_downtime = Downtime(downtime_list[0].host_name, new_end_time, comment_string, \
        downtime_list[0].service, "00000")
    new_downtime_id = merged_downtime.new_downtime()

    #this loop will work fine as it will remove all items (Downtimes) in the list.
    #the newly created Downtime will remain as it is not in the original list
    #NOTE: This remove_downtime() function only works for HOST
    for old_downtime in downtime_list:
        old_downtime.remove_downtime()

    return merged_downtime

############## CONNECT TO THINGS ##############
def connect_to_livestatus(nagios_cmd_string):
    #this opens up the nagios.cmd socket and writes the command to it, 'with:' closes when it's done.
    try:
        with open(NAGIOS_CMD_LOCATION, 'w') as nagios_cmd:
            nagios_cmd.writelines("{}".format(nagios_cmd_string))
            LOGGER.info(nagios_cmd_string)
    except IOError as e:
        LOGGER.crit("Cannot write to Nagios.cmd!!")
    return

def get_downtimes():
    """This hits the livestatus lql call for getting all current downtimes"""
    retry_socket_attempt = 1
    yaml_answer = None
    message = 'GET downtimes\n'
    message += 'Columns: host_name end_time comment service_display_name id\n'
    message += 'OutputFormat: json\n\n'
    answer = b''
    #this is gross, i know, but livestatus doesnt always respond perfectly
    #gives us a little more chance to not fail and annoy people
    while retry_socket_attempt < 5:
        try:
            #build the LQL query, to find columns, query like this check 1st row of a blank query
            nagios_socket_connection = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
            nagios_socket_connection.connect(NAGIOS_LIVE_SOCK)
            #send the message, shutdown the socket's messaging. Need to close?
            nagios_socket_connection.send(message)
            # Now read the answer
            while 1:
                response = nagios_socket_connection.recv(1000000)
                if not response:
                    break
                else:
                    answer += response

            nagios_socket_connection.shutdown(socket.SHUT_WR)
            yaml_answer = yaml.load(answer)

            nagios_socket_connection.close()
            LOGGER.info("Found {} downtimes on this video-nagios host.".format(len(yaml_answer)))
        except (socket.error, os.error) as e:
            LOGGER.critical("LiveStatus Socket System Error: {0}".format(e))
        except socket.timeout as e:
            LOGGER.critical("LiveStatus Socket Timeout: {0}".format(e))
        except yaml.YAMLError as e:
            LOGGER.error("Error loading YAML from LiveStatus: {0}".format(e))

        if yaml_answer:
            break
        else:
            retry_socket_attempt = retry_socket_attempt + 1

    #spit the answer back in YAML
    return yaml_answer

def connect_to_jira():
    """Connects to JIRA as user video-nagios"""

    jira_options = {'server': "https://jira.xarth.tv"}
    jira_key_location = '/etc/nagios/video/video-nagios-private.pem'
    jira_token_location = '/etc/nagios/video/jira-token.auth'

    try:
        with open(jira_key_location, 'r') as key_cert_file:
            key_cert_data = key_cert_file.read()
        with open(jira_token_location, 'r') as credentials_file:
            tokens = credentials_file.readline()
            access_token = tokens.split(':')[0]
            access_token_secret = tokens.split(':')[1]
            #avoid the new line char

        # This contains tokens generated by jirashell
        #jirashell -pt -s 'https://twitchtv.atlassian.net' -u video-nagios -p password -od -ck "videonagios2" -k video-nagios-private.pem)
        oauth_dict = {
            'access_token': access_token,
            'access_token_secret': access_token_secret,
            'consumer_key': 'videonagios2',
            'key_cert': key_cert_data
        }

        jira_connection = jira.JIRA(options=jira_options, oauth=oauth_dict, validate=True, logging=False)
        LOGGER.info("Connection to JIRA successful")
    except (jira.exceptions.JIRAError, exceptions.ConnectionError) as e:
        LOGGER.critical("Cannot connect to JIRA: {0}".format(str(e)))
        jira_connection = False
        LOGGER.warn("Exiting the script as the connection to JIRA failed")
        sys.exit()
    return jira_connection

if __name__ == '__main__':
    """actually do the things"""
    list_of_downtimes = get_downtimes()
    sorted_dict_of_downtimes = sort_downtimes(list_of_downtimes)
    iterate_through_downtimes(sorted_dict_of_downtimes)
