#!/usr/bin/python
import argparse
import json
import logging
from logging.handlers import SysLogHandler
import os
import uuid
import datetime
import requests
from requests.auth import AuthBase

LOGGER = logging.getLogger('syslog')
formatter = logging.Formatter("notif_to_alerta [%(process)s]: %(levelname)s - %(message)s")
handler = SysLogHandler('/dev/log', facility=logging.handlers.SysLogHandler.LOG_LOCAL3)
handler.setFormatter(formatter)
LOGGER.addHandler(handler)

ALERTA_ENDPOINT = 'https://video-alerta.us-west-2.staging.alerta.live-video.a2z.com/api/alert'
ALERTA_KEY      = 'KubpGl7yGIVMSuz9PFoaiZvoN-c7wFuK5tS_pak4'

def handle_arguments():
    """handle arguments from the nagios command, build subject/body for notification"""
    parser = argparse.ArgumentParser()
    parser.add_argument("--alert_state", action="store")
    parser.add_argument("--hostname", action="store")
    parser.add_argument("--output", action="store")
    parser.add_argument("--alert_date", action="store")
    parser.add_argument("--service_desc", action="store")
    parser.add_argument("--pagerduty_secret", action="store")
    args = parser.parse_args()

    #hands back a nagiosnotif object and the proxy argument
    return NagiosNotification(args.alert_state, args.hostname, args.output,\
        args.alert_date, args.service_desc, args.pagerduty_secret)

class CustomJsonEncoder(json.JSONEncoder):
    def default(self, o):
        if isinstance(o, (datetime.date, datetime.datetime)):
            return o.replace(microsecond=0).strftime('%Y-%m-%dT%H:%M:%S') + '.%03dZ' % (o.microsecond // 1000)
        elif isinstance(o, datetime.timedelta):
            return int(o.total_seconds())
        else:
            return json.JSONEncoder.default(self, o)

class ApiKeyAuth(AuthBase):

    def __init__(self, api_key=None, auth_token=None):
        self.api_key = api_key
        self.auth_token = auth_token

    def __call__(self, r):
        r.headers['Authorization'] = 'Key {}'.format(self.api_key)
        return r

class NagiosNotification(object):
    def __init__(self, alert_state, alert_host, alert_output, alert_date, \
        alert_service, pagerduty_secret):
        self.alert_state = alert_state
        self.alert_host = alert_host
        self.alert_output = alert_output
        self.alert_date = alert_date
        self.alert_service = alert_service
        self.pagerduty_secret = pagerduty_secret

    def alert_pop(self):
        """ Cut hostname down to just pop """
        host_name_split = self.alert_host.split('.')
        if len(host_name_split) > 1:
            pop_name = host_name_split[1]
        else:
            pop_name = self.alert_host
        return pop_name

    def alert_cluster(self):
        """ Cut hostname to grab clustername """
        cluster_name = ""
        alert_host = self.alert_host
        if '.' in alert_host:
            server_name = alert_host.split('.')[0]
            if "-" in server_name:
                for cluster_string in server_name.split('-')[:-1]:
                    cluster_name += cluster_string + "-"
                cluster_name = cluster_name[:-1]
        else:
            cluster_name = alert_host
        return cluster_name

    def notif_message(self):
        """Convert host notifications into Alerta criticality level"""
        if self.alert_service.lower() == "host":
          if self.alert_state.lower() == "up":
            severity = "ok"
          else:
            severity = "critical"
        else:
          severity = self.alert_state.lower()
        """Create message to go into queue"""
        alert_body = {
            'resource': self.alert_host,
            'event': self.alert_service,
            'environment': 'Production',
            'severity': severity,
            'correlate': [],
            'service': [self.alert_service],
            'group': self.alert_cluster() + '-' + self.alert_pop(),
            'value': self.alert_state,
            'text': self.alert_output,
            'tags': [],
            'attributes': {
                'pagerduty': self.pagerduty_secret,
                'date': self.alert_date
            },
            'origin': self.alert_pop(),
            'type': 'Nagios',
            'createTime': datetime.datetime.now()
        }
        return alert_body

def push_to_alerta(notif):
    """check connection to cache and push notif"""
    #push item to specific list for specific cache
    session = requests.Session()
    session.trust_env = False
    session.headers.update({
        'X-Request-ID': str(uuid.uuid4()),
        'Content-type': 'application/json'
    })
    try:
        LOGGER.info("Sending Nagios event to Alerta: " + str(notif.notif_message()))
        response = session.post(ALERTA_ENDPOINT, data=json.dumps(notif.notif_message(), cls=CustomJsonEncoder), auth=ApiKeyAuth(ALERTA_KEY), timeout=30)
        if response.status_code == 201:
          LOGGER.info("Alert created: " + str(response.json()))
        elif response.status_code == 202:
          LOGGER.info("Alert accepted (blackout): " + str(response.json()))
        else:
          LOGGER.critical("Error response received: " + str(response.json()))
    except requests.exceptions.RequestException as e:
        #unexpected
        LOGGER.critical("Unexpected Error: " + str(e))

if __name__ == "__main__":
    nagios_alert = handle_arguments()
    push_to_alerta(nagios_alert)
