#!/usr/bin/python
import argparse
import json
import boto3
import botocore
import logging
from logging.handlers import SysLogHandler
import os

LOGGER = logging.getLogger('syslog')
formatter = logging.Formatter("notif_to_kinesis [%(process)s]: %(levelname)s - %(message)s")
handler = SysLogHandler('/dev/log', facility=logging.handlers.SysLogHandler.LOG_LOCAL3)
handler.setFormatter(formatter)
LOGGER.addHandler(handler)

def handle_arguments():
    """handle arguments from the nagios command, build subject/body for notification"""
    parser = argparse.ArgumentParser()
    parser.add_argument("--alert_state", action="store")
    parser.add_argument("--hostname", action="store")
    parser.add_argument("--output", action="store")
    parser.add_argument("--alert_date", action="store")
    parser.add_argument("--service_desc", action="store")
    parser.add_argument("--stream_id", action="store")
    parser.add_argument("--pagerduty_secret", action="store")
    parser.add_argument("--source_host", action="store")
    parser.add_argument("--notes", action="store")
    args = parser.parse_args()

    #hands back a nagiosnotif object and the proxy argument
    return NagiosNotification(args.alert_state, args.hostname, args.output,\
        args.alert_date, args.service_desc, args.pagerduty_secret, args.source_host, args.notes), args.stream_id

class NagiosNotification(object):
    def __init__(self, alert_state, alert_host, alert_output, alert_date, \
        alert_service, pagerduty_secret, source_host, notes):
        self.alert_state = alert_state
        self.alert_host = alert_host
        self.alert_output = alert_output
        self.alert_date = alert_date
        self.alert_service = alert_service
        self.pagerduty_secret = pagerduty_secret
        self.source_host = source_host
        self.notes = notes

    def alert_pop(self):
        """ Cut hostname down to just pop """
        host_name_split = self.alert_host.split('.')
        if len(host_name_split) > 1:
            pop_name = host_name_split[1]
        else:
            pop_name = self.alert_host
        return pop_name

    def alert_cluster(self):
        """ Cut hostname to grab clustername """
        cluster_name = ""
        alert_host = self.alert_host
        if '.' in alert_host:
            server_name = alert_host.split('.')[0]
            if "-" in server_name:
                for cluster_string in server_name.split('-')[:-1]:
                    cluster_name += cluster_string + "-"
                cluster_name = cluster_name[:-1]
        else:
            cluster_name = alert_host
        return cluster_name

    def notif_message(self):
        """Create message to go into queue"""
        alert_body = json.dumps({
            'STATE': self.alert_state,
            'HOST': self.alert_host,
            'CLUSTER': self.alert_cluster(),
            'POP': self.alert_pop(),
            'SERVICE': self.alert_service,
            'OUTPUT': self.alert_output,
            'ALERTDATE': self.alert_date,
            'PAGERDUTY': self.pagerduty_secret,
            'SOURCE' : self.source_host,
            'NOTES' : self.notes
        })
        return alert_body

def push_to_kinesis(notif, stream_id):
    """check connection to cache and push notif"""
    #push item to specific list for specific cache
    try:
        boto3.setup_default_session(profile_name='default')     
        kinesis_client = boto3.client('kinesis')
        kinesis_client.put_record(StreamName=stream_id, Data=notif.notif_message(), PartitionKey='1')
        LOGGER.critical("Payload has been input into Kinesis: {0}".format(str(notif.notif_message())))
    except botocore.exceptions.EndpointConnectionError as e:
        #cant connect
        LOGGER.critical("Cannot connect to Kinesis!! : {0}".format(str(e)))
    except botocore.exceptions.ClientError as e:
        if e.response['Error']['Code'] == 'ResourceNotFoundException':
            #no data sent
            LOGGER.critical("Null Data or Incorrect Data format: {0}".format(str(e)))
        else:
            #unexpected
            LOGGER.critical("Unexpected Error: {0}".format(str(e)))
            raise 


if __name__ == "__main__":
    nagios_alert, stream_id = handle_arguments()
    push_to_kinesis(nagios_alert, stream_id)
