# Install NRDP. To make this work, simply set a token in hiera:
# twitch_nagios::nrdp::token: some_long_random_string_here

class twitch_nagios::nrdp (
  $tokens = [],
) inherits twitch_nagios {
  $_vhost_name      = $twitch_nagios::vhost_name
  $_auto_make       = $twitch_nagios::auto_make
  $_install_nagios  = $twitch_nagios::install_nagios
  $nrdp_tokens      = $twitch_nagios::nrdp::tokens
  $nrdp_dir         = $twitch_nagios::nrdp_dir
  $repo_path        = $twitch_nagios::repo_path

  include twitch_nrdp
  
  twitch_nrdp::check { 'puppet-agent':
     command        => '/usr/lib/nagios/plugins/check_file_age -w 6600 -c 13200 -f /var/lib/puppet/state/last_run_summary.yaml',
     interval       => 4,
     retry          => 1,
     retry_interval => 2,
  }
  
  case $_auto_make {
    true: { $router_file = '/etc/nagios/router/nrdp-router.yaml' }
    false: { $router_file = "/etc/nagios/static/nrdp-router.yaml" }
    default: {
      fail('auto_make must be a boolean')
    }
  }

  case $_install_nagios {
    true: { $_nrdp_index_path = 'twitch_nagios/nrdp/index.php.erb' }
    false: { $_nrdp_index_path = 'twitch_nagios/nrdp/nrdp_forwarder.php.erb' }
    default: {
      fail('install_nagios must be a boolean')
    }
  }

  if empty($tokens) {
    $ensure = 'absent'
    $dir_ensure = 'absent'
    $ensure_alt_php = 'absent'
  } else {
    $ensure = 'present'
    $dir_ensure = 'directory'
    # Add www-data to the nagcmd groups. This allows NRDP/Apache to talk to Nagios.
    User <| title == $::apache::user |> {
      groups  +> $twitch_nagios::install::nagios_command_group,
    }
    $ensure_alt_php = $twitch_nagios::ensure_alt_php ? {
      absent  => 'absent',
      default => 'file',
    }
  }
  file { $nrdp_dir:
    ensure => $dir_ensure,
    owner  => 'root',
    group  => 'root',
    mode   => '0755',
    force  => true,
  }
  file { "${nrdp_dir}/server":
    ensure => $dir_ensure,
    owner  => 'root',
    group  => 'root',
    mode   => '0755',
    force  => true,
  }

  # Write a config file with a list of all Nagios servers
  # For use by NRDP when forwarding checks
  file { "${nrdp_dir}/server/nagios-servers.cfg.ctmpl":
    ensure  => present,
    owner   => 'root',
    mode    => '0644',
    content => template('twitch_nagios/nagios-servers.cfg.ctmpl.erb'),
    require => File["${nrdp_dir}/server"],
  }
  consul_template::service{ 'nrdp_list':
    ensure      => present,
    source      => "${nrdp_dir}/server/nagios-servers.cfg.ctmpl",
    destination => "${nrdp_dir}/server/nagios-servers.cfg",
    command     => '/bin/true',
    require     => File["${nrdp_dir}/server/nagios-servers.cfg.ctmpl"]
  }

  # Replace the Thruk livestatus interface with a command file script instead
  # This eliminates crashing on large Thruk installations (video-nagios).
  # In nrdp.pp because it requires nrdp extensions to be installed to work.
  file { '/usr/share/thruk/.htaccess':
    ensure  => $ensure_alt_php,
    mode    => '0644',
    owner   => 'root',
    group   => 'root',
    source  => 'puppet:///modules/twitch_nagios/thruk_alt_php_htaccess',
    require => Package['thruk'],
  }

  # This lives outside the below if stanza so calls to this concat while nrdp
  # is disabled will not cause manifest compile errors. Some fragments below.
  concat { "${nrdp_dir}/server/config.inc.php":
    ensure => $ensure,
    owner  => $::apache::user,
    group  => $::apache::group,
    mode   => '0440',
  }

  apache::custom_config { 'nrdp':
    ensure  => $ensure,
    content => template('twitch_nagios/nrdp/apache.conf.erb'),
  }

  if $dir_ensure != 'absent' {
    file { "${nrdp_dir}/server/index.php":
      ensure  => 'present',
      owner   => 'root',
      content => template($_nrdp_index_path),
      require => File["${nrdp_dir}/server"],
      before  => Apache::Custom_config['nrdp'],
    }
    -> file { "${nrdp_dir}/server/spyc.php":
      ensure  => 'present',
      owner   => 'root',
      source  => 'puppet:///modules/twitch_nagios/spyc.php',
      require => File["${nrdp_dir}/server"],
      before  => Apache::Custom_config['nrdp'],
    }
    -> file { "${nrdp_dir}/server/includes":
      ensure  => 'present',
      recurse => true,
      replace => true,
      owner   => 'root',
      source  => 'puppet:///modules/twitch_nagios/nrdp/includes',
    }
    -> file { "${nrdp_dir}/server/plugins":
      ensure  => 'present',
      recurse => true,
      replace => true,
      owner   => 'root',
      source  => 'puppet:///modules/twitch_nagios/nrdp/plugins',
    }
    # This file is not actually part of nrdp, but it lives in the nrdp_dir folder.
    # It is only used when twitch_nagios::ensure_alt_php = present or file.
    file { "${nrdp_dir}/server/command.php":
      ensure  => 'present',
      owner   => 'root',
      group   => 'www-data',
      mode    => '0644',
      content => template('twitch_nagios/nrdp/command.php.erb'),
    }
    # This is a template so we can control the token with hiera.
    concat::fragment { 'config.inc.php_main':
      target  => "${nrdp_dir}/server/config.inc.php",
      content => template('twitch_nagios/nrdp/config.inc.php.erb'),
      order   => '10',
    }
    # NCGP uses this if it's enabled.
    concat::fragment { 'config.inc.php_ncgp':
      target  => "${nrdp_dir}/server/config.inc.php",
      content => '//',
      order   => '80',
    }
    concat::fragment { 'config.inc.php_footer':
      target  => "${nrdp_dir}/server/config.inc.php",
      content => "\n\n?>\n",
      order   => '90',
    }
  }
}
