# Checks out the latest version of the static Nagios configuration from Git
class twitch_nagios::repository (
  $_auto_make            = $twitch_nagios::auto_make,
  $_install_nagios       = $twitch_nagios::install_nagios,
  $secret_name           = "${::twitch_team}/${twitch_nagios::server_name}-nagios/${::twitch_environment}/nagios-git-ssh-private-key",
  $repo_path             = $twitch_nagios::repo_path,
  $router_path           = '/etc/nagios/router',
  $shards_path           = '/var/nagios/cache/shards',
  $shards_tmp_path       = '/var/nagios/cache/shards_tmp',
  $url                   = "systems/${twitch_nagios::server_name}-nagios",
  $concurrent_checks     = 300,
  $check_result_path     = '/var/nagios/spool/checkresults',
  $command_file          = '/var/nagios/rw/nagios.cmd',
  $branch                = 'master',
) inherits twitch_nagios {

  $git_repo_url = "${twitch_nagios::git_server}:${url}"

  case $_auto_make {
    true: {
      $output_path     = "${shards_path}/${::fqdn}"
      $automake_ensure = 'present'

    }
    false: {
      $output_path     = "${repo_path}/shards/${::fqdn}"
      $automake_ensure = 'absent'
    }
    default: {
      fail('auto_make must be a boolean')
    }
  }

  file { $repo_path:
    ensure => 'directory',
    owner  => $twitch_nagios::nagios_user,
    group  => $twitch_nagios::nagios_group,
    mode   => '0755',
  }

  # Location of the Router Config file
  file { $router_path:
    ensure => 'directory',
    owner  => $twitch_nagios::nagios_user,
    group  => $twitch_nagios::nagios_group,
    mode   => '0755',
  }

  # Location of the Staged Nagios Config files
  file { $shards_tmp_path:
    ensure => 'directory',
    owner  => $twitch_nagios::nagios_user,
    group  => $twitch_nagios::nagios_group,
    mode   => '0755',
  }

  # Location of the Dynamic Nagios Config files
  file { $shards_path:
    ensure => 'directory',
    owner  => $twitch_nagios::nagios_user,
    group  => $twitch_nagios::nagios_group,
    mode   => '0755',
  }

  file { '/var/lib/nagios/.ssh':
    ensure => 'directory',
    owner  => $twitch_nagios::nagios_user,
    group  => $twitch_nagios::nagios_group,
    mode   => '0700',
  }

  twitch_sandstorm_agent::template { 'nagios-git-ssh-private-key':
    destination     => '/var/lib/nagios/.ssh/id_rsa',
    owner           => $twitch_nagios::nagios_user,
    group           => $twitch_nagios::nagios_group,
    mode            => '0600',
    contents        => "{{ key \"${secret_name}\" }}\n",
    restart_command => '',
    require         => File['/var/lib/nagios/.ssh/', $repo_path, $router_path, $shards_path, $shards_tmp_path],
  }

  # Install Auto Make Helper
  file { '/var/lib/nagios/auto_make.sh':
    ensure  => $automake_ensure,
    owner   => 'root',
    group   => 'root',
    mode    => '0755',
    content => template('twitch_nagios/auto_make.sh.erb'),
  }

  cron::job { 'nagios_auto_make':
    ensure  => $automake_ensure,
    command => '/var/lib/nagios/auto_make.sh  2>&1 | logger -p local3.info -t auto_make',
    user    => $twitch_nagios::nagios_user,
    minute  => '*/15',
    require => File['/var/lib/nagios/auto_make.sh'],
  }

  # Install sudo permissions for automake
  sudo::conf { 'ttv-nagios_auto_make':
    priority => 99,
    source   => "puppet:///modules/${module_name}/sudoers.nagios",
  }

  if $_install_nagios {
    file { '/etc/nagios/cgi.cfg':
      ensure => 'link',
      target => "${repo_path}/cgi.cfg",
      owner  => $twitch_nagios::nagios_user,
      group  => $twitch_nagios::nagios_group,
    }

    file { '/etc/nagios/nagios.cfg':
      ensure  => 'file',
      content => template('twitch_nagios/nagios.cfg.erb'),
      owner   => $twitch_nagios::nagios_user,
      group   => $twitch_nagios::nagios_group,
      notify  => Service['nagios'],
    }

    file { ['/etc/nagios/resource.cfg', '/etc/nagios/nrpe_local.cfg']:
      ensure => 'absent',
    }

    vcsrepo { $repo_path:
      ensure   => 'latest',
      revision => $branch,
      provider => 'git',
      source   => "git@${git_repo_url}.git",
      user     => $twitch_nagios::nagios_user,
      require  => Twitch_sandstorm_agent::Template['nagios-git-ssh-private-key'],
      notify   => Service['nagios'],
    }

    # Ensure backwards compatibility until we are sure nothing is hardcoded
    file { '/etc/nagios/static':
      ensure  => 'link',
      target  => $repo_path,
      require => Mount['/var/nagios/cache'],
    }

  } else {
    # Don't notify Service['nagios'] if it is not installed
    vcsrepo { $repo_path:
      ensure   => 'latest',
      revision => $branch,
      provider => 'git',
      source   => "git@${git_repo_url}.git",
      user     => $twitch_nagios::nagios_user,
      require  => Twitch_sandstorm_agent::Template['nagios-git-ssh-private-key'],
    }

    file { '/etc/nagios/static':
      ensure => 'directory',
    }
  }
}
