# Configures Nagios vhost
class twitch_nagios::vhost (
  $sandstorm  = true,
) inherits twitch_nagios {
  $env         = $twitch_nagios::env
  $_vhost_name = $twitch_nagios::vhost_name
  $ldap_server = $twitch_nagios::ldap_server
  $ldap_config = {
    auth_require        => 'valid-user',
    auth_basic_provider => 'ldap',
    auth_type           => 'basic',
    auth_name           => 'Nagios [LDAP]',
    custom_fragment     => "AuthLDAPURL ldaps://${ldap_server}/ou=Users,dc=justin,dc=tv",
  }

  if $sandstorm {
    $cert_file = '/etc/ssl/certs/wildcard.internal.justin.tv.pem'
    $key_file  = '/etc/ssl/private/wildcard.internal.justin.tv.key'
    $wildcard_jtv_cert = join([
      "{{ key \"video/ssl_keys/${env}/wildcard_internal_justin_tv/cert\" }}",
      "{{ key \"video/ssl_keys/${env}/wildcard_internal_justin_tv/intermediate\" }}",
    ], "\n")
    $wildcard_jtv_key  = "video/ssl_keys/${env}/wildcard_internal_justin_tv/key"

    twitch_ssl_certificate::certificate { 'wildcard-jtv-cert':  # name here was the name of the old twitch_sandstorm_agent::template for the cert
      cert_filesystem_path => $cert_file,
      cert_contents        => $wildcard_jtv_cert,
      cert_restart_cmd     => '/usr/sbin/service apache2 restart',
      key_identifier       => 'wildcard-jtv-key', # this was the name of the old twitch_sandstorm_agent::template for the cert
      key_filesystem_path  => $key_file,
      key_contents         => "{{ key \"${wildcard_jtv_key}\" }}",
      key_mode             => '0644',
      key_restart_cmd      => '/usr/sbin/service apache2 restart',
    }

  } else {
    $cert_file = "/etc/ssl/certs/${_vhost_name}.pem"
    $key_file  = "/etc/ssl/private/${_vhost_name}.key"
    # For use behind ELB/ALB using AWS ACM cert.
    exec { "self-signed cert ${_vhost_name}":
      command => join([
        'openssl req -new -newkey rsa:2048 -days 1095 -nodes -x509',
        '-keyout', $key_file, '-out', $cert_file,
        "-subj '/C=US/ST=California/L=San Francisco/CN=${_vhost_name}'",
      ], ' '),
      user    => root,
      path    => ['/usr/bin','/bin'],
      creates => [$key_file, $cert_file],
      before  => Class['apache'],
    }
}
  apache::vhost { 'nagios-redir':
    servername => $_vhost_name,
    port       => 80,
    docroot    => '/var/www/redirect',
    rewrites   => [
      {
        rewrite_rule => ['^/server-status$ - [L]', "(.*) https://${_vhost_name}/ [R=301,L]"]
      }
    ]
  }

  apache::vhost { 'nagios':
    servername           => $_vhost_name,
    port                 => '443',
    docroot              => '/var/www',
    ssl                  => true,
    ssl_cert             => $cert_file,
    ssl_chain            => $cert_file,
    ssl_key              => $key_file,
    redirectmatch_status => '301',
    redirectmatch_regexp => '^/$',
    redirectmatch_dest   => '/thruk/',
    aliases              => [
      {
        scriptalias => '/nagios/cgi-bin',
        path        => '/usr/share/nagios/cgi',
      },
      {
        alias => '/nagios',
        path  => '/usr/share/nagios/web',
      },
      {
        alias => '/images',
        path  => '/usr/share/nagios/web/images',
      },
      {
        alias => '/js',
        path  => '/usr/share/nagios/web/js',
      },
      {
        alias => '/stylesheets',
        path  => '/usr/share/nagios/web/stylesheets',
      },
    ],
    directories          => [
      merge({
        path           => '/usr/share/nagios/web',
        options        => ['None'],
        allow_override => ['None'],
        order          => 'allow,deny',
        require        => 'all granted',
      }, $ldap_config),
      merge({
        path           => '/usr/share/nagios/cgi',
        allow_override => 'None',
        order          => 'allow,deny',
        require        => 'all granted',
      }, $ldap_config),
    ],
  }

  class {'apache::mod::authnz_ldap':
    verify_server_cert => false,
  }

  class { 'apache::mod::status':
      allow_from      => ['127.0.0.1','::1', '10.0.0.0/8'],
      extended_status => 'On',
      status_path     => '/server-status'
  }

  # "health check"s for Midway's usage
  # AMZN standard is to use /ping and /sping
  # Don't put anything cute in here; it's not protected by auth
  file {['/var/www/ping', '/var/www/sping']:
    content => 'ok',
    require => Class['apache'],
    mode    => '0644',
  }

}
