#!/bin/bash
#

facter boardmanufacturer manufacturer 2>/dev/null | grep -qi kontron || {
    echo Kontron hardware not found. exiting
    exit 0
}

# ipmi-sensors from freeipmi package does not work with Kontron BMCs; use ipmitool instead
# payload number is determined by B1 or B2 in the sensor prefix
#   ipmitool sdr list
#     B1:IPMI Info-1   | 0x00              | ok
#

type ipmitool >/dev/null 2>&1 || {
    echo >&2 ipmitool command not found
    exit 127
}

set -e -u

# Verified kontron by this point. Load ipmi_si kernel module otherwised blacklist from
# loading automatically
modprobe ipmi_si

payload_num=$(ipmitool sdr list | head -n1 | cut -c2)

case $payload_num in
    1) local_addr=0x7a ;;
    2) local_addr=0x7c ;;
    *) echo >&2 Payload number not found && exit 1 ;;
esac

# query primary ShMC for Chassis serial number
chassis_serial=$(ipmitool -m${local_addr} -T0x20 -t0x20 fru print | awk '/Chassis Serial/ {print $4; exit}')
# query local vMMC for Board serial number
board_serial=$(ipmitool fru print | awk '/Board Serial/ {print $4; exit}')
# query local Node BMC for lan information
output=$(ipmitool -m${local_addr} -t0x20 lan print)
ipaddress_ipmi=$(echo "$output" | awk '/IP Address *:/ {print $4; exit}')
macaddress_ipmi=$(echo "$output" | awk '/MAC Address *:/ {print $4}')

facter_dir=/etc/facter/facts.d
[ -d "$facter_dir" ] || exit 1

# check all variables for existence
for var in chassis_serial board_serial ipaddress_ipmi macaddress_ipmi; do
    [ -z "${!var}" ] && echo >&2 "Unable to determine $var, exiting." && exit 1
done

echo "chassisserialnumber=$chassis_serial" > ${facter_dir}/chassisserialnumber.txt
echo "servernumber=$payload_num" > ${facter_dir}/servernumber.txt
echo "ipaddress_ipmi=$ipaddress_ipmi" > ${facter_dir}/ipaddress_ipmi.txt
echo "macaddress_ipmi=$macaddress_ipmi" > ${facter_dir}/macaddress_ipmi.txt
