# Nydus is the video replication HTTP cache server with TLS support
class twitch_nydus (
  $ensure                    = $twitch_nydus::params::ensure,
  $env                       = $twitch_nydus::params::env,
  $port                      = $twitch_nydus::params::port,
  $tls_port                  = $twitch_nydus::params::tls_port,
  $internal_addr             = $twitch_nydus::params::internal_addr,
  $statsd_host               = $twitch_nydus::params::statsd_host,
  $statsd_port               = $twitch_nydus::params::statsd_port,
  $statsd_addr               = $twitch_nydus::params::statsd_addr,
  $statsd_rate               = $twitch_nydus::params::statsd_rate,
  $ssl_alg                   = $twitch_nydus::params::ssl_alg,
  $vitreous_url              = $twitch_nydus::params::vitreous_url,
  $systemd_start_timeout_sec = $twitch_nydus::params::systemd_start_timeout_sec,
  $binary                    = $twitch_nydus::params::binary,
  $memory_policy_interleave  = $twitch_nydus::params::memory_policy_interleave
) inherits twitch_nydus::params {

  validate_string($env, $statsd_host, $statsd_addr, $vitreous_url, $ensure, $binary)
  validate_integer($port)
  validate_re($binary, '^nydus_(edge|pr|ads|cdn)$',
    'The value for binary must be either "nydus_edge", "nydus_pr", "nydus_cdn", or "nydus_ads"')

  $addr = ":${port}"

  if $env == 'canary' {
    $consul_tags = ['canary', 'production', "${binary}_canary", "${binary}_production"]
  } else {
    $consul_tags = [$env, "${binary}_${env}"]
  }

  $user = 'nydus'
  twitch::local_user { $user:
    group => $user,
  }

  # Monitoring
  twitch_servicecheck::passive { 'HTTP Backend Health':
    command        => "check_http -H 127.0.0.1 -u /health -p ${port}",
    interval       => 1, # minutes between checks
    retry          => 2, # times to fail before reporting failure
    retry_interval => 1, # minutes between checks once reporting failure
  }

  # Configuration
  $conf_dir = '/etc/nydus'
  file { 'nydus_dir':
    ensure  => directory,
    path    => $conf_dir,
    owner   => $user,
    group   => $user,
    require => Twitch::Local_user[$user],
  }

  $autoprof_bucket = $env ? {
    'staging'    => 'twitch-video-replication-prism-autoprof-dev',
    'production' => 'twitch-video-replication-prism-autoprof-prod',
    'canary'     => 'twitch-video-replication-prism-autoprof-prod',
    default      => undef,
  }

  # SSL (if needed)
  if $tls_port {
    validate_integer($tls_port)
    $tls_addr = ":${tls_port}"
    $ssl_dir = "${conf_dir}/ssl"

    file { 'nydus_ssl_dir':
      ensure  => directory,
      path    => $ssl_dir,
      owner   => $user,
      group   => $user,
      require => File['nydus_dir'],
    }

    twitch_servicecheck::passive { 'HTTPS Backend Health':
      command        => "check_http --ssl -H 127.0.0.1 -u /health -p ${tls_port}",
      interval       => 1, # minutes between checks
      retry          => 2, # times to fail before reporting failure
      retry_interval => 1, # minutes between checks once reporting failure
    }

    # Default parameters for all certificates
    Twitch_nydus::Ssl {
      user     => $user,
      alg_type => $ssl_alg,
      dir      => $ssl_dir,
      require  => File['nydus_ssl_dir'],
    }

    # Special case cdn, so we don't roll every cert out everywhere.
    # TODO: properly refactor this to reduce branching here.
    if $binary == 'nydus_cdn' {

      # *.cdn-origin.hls.live-video.net
      twitch_nydus::ssl { 'cdn_origin':
        sandstorm_key_prefix => 'video/cdn-origin/production/wildcard_cdn-origin_hls_live-video_net',
      }

      $ecdsa_certs = [
        { 'Cert' => "${ssl_dir}/cdn_origin_ecdsa_cert.pem", 'Key' => "${ssl_dir}/cdn_origin_ecdsa_key.pem" },
      ]

      $rsa_certs = [
        { 'Cert' => "${ssl_dir}/cdn_origin_rsa_cert.pem", 'Key' => "${ssl_dir}/cdn_origin_rsa_key.pem" },
      ]

      # Puppet 3.8 Doesn't allow array concatenation..
      $both_certs = [
        { 'Cert' => "${ssl_dir}/cdn_origin_ecdsa_cert.pem", 'Key' => "${ssl_dir}/cdn_origin_ecdsa_key.pem" },
        { 'Cert' => "${ssl_dir}/cdn_origin_rsa_cert.pem", 'Key' => "${ssl_dir}/cdn_origin_rsa_key.pem" },
      ]

    } else {

      # *.<pop>.abs.hls.ttvnw.net
      twitch_nydus::ssl { 'pop_abs':
        sandstorm_key_prefix => "video/video-edge/production/${::pop}/wildcard_abs_hls_ttvnw_net",
      }

      # *.<pop>.no-abs.hls.ttvnw.net
      twitch_nydus::ssl { 'pop_no-abs':
        sandstorm_key_prefix => "video/video-edge/production/${::pop}/wildcard_no-abs_hls_ttvnw_net",
      }

      # TODO: programmatically calculate these
      $ecdsa_certs = [
        { 'Cert' => "${ssl_dir}/pop_abs_ecdsa_cert.pem", 'Key' => "${ssl_dir}/pop_abs_ecdsa_key.pem" },
        { 'Cert' => "${ssl_dir}/pop_no-abs_ecdsa_cert.pem", 'Key' => "${ssl_dir}/pop_no-abs_ecdsa_key.pem" },
      ]

      $rsa_certs = [
        { 'Cert' => "${ssl_dir}/pop_ssl_rsa_cert.pem", 'Key' => "${ssl_dir}/pop_ssl_rsa_key.pem" },
        { 'Cert' => "${ssl_dir}/pop_ivs_rsa_cert.pem", 'Key' => "${ssl_dir}/pop_ivs_rsa_key.pem" },
        { 'Cert' => "${ssl_dir}/pop_abs_rsa_cert.pem", 'Key' => "${ssl_dir}/pop_abs_rsa_key.pem" },
        { 'Cert' => "${ssl_dir}/pop_no-abs_rsa_cert.pem", 'Key' => "${ssl_dir}/pop_no-abs_rsa_key.pem" },
      ]
      # Puppet 3.8 Doesn't allow array concatenation..
      $both_certs = [
        { 'Cert' => "${ssl_dir}/pop_abs_ecdsa_cert.pem", 'Key' => "${ssl_dir}/pop_abs_ecdsa_key.pem" },
        { 'Cert' => "${ssl_dir}/pop_no-abs_ecdsa_cert.pem", 'Key' => "${ssl_dir}/pop_no-abs_ecdsa_key.pem" },
        { 'Cert' => "${ssl_dir}/pop_ssl_rsa_cert.pem", 'Key' => "${ssl_dir}/pop_ssl_rsa_key.pem" },
        { 'Cert' => "${ssl_dir}/pop_ivs_rsa_cert.pem", 'Key' => "${ssl_dir}/pop_ivs_rsa_key.pem" },
        { 'Cert' => "${ssl_dir}/pop_abs_rsa_cert.pem", 'Key' => "${ssl_dir}/pop_abs_rsa_key.pem" },
        { 'Cert' => "${ssl_dir}/pop_no-abs_rsa_cert.pem", 'Key' => "${ssl_dir}/pop_no-abs_rsa_key.pem" },
      ]

      # *.<pop>.hls.ttvnw.net
      twitch_nydus::ssl { 'pop_ssl':
        sandstorm_key_prefix => "video/video-edge/production/${::pop}/wildcard_hls_ttvnw_net",
        alg_type             => 'rsa',
      }

      # *.<pop>.hls.live-video.net
      twitch_nydus::ssl { 'pop_ivs':
        sandstorm_key_prefix => "video/video-edge/production/${::pop}/wildcard_hls_live-video_net",
        alg_type             => 'rsa',
      }
    }

    if $ssl_alg == 'rsa' {
      $tls_certs = $rsa_certs
    } elsif $ssl_alg == 'ecdsa' {
      $tls_certs = $ecdsa_certs
    } else {
      $tls_certs = $both_certs
    }

  }

  # Run settings
  $pidfile = '/run/nydus/pid'
  $optional_process_wrapper = $memory_policy_interleave ? {
    true => ['/usr/bin/numactl', '--interleave=all', '--'],
    default => undef,
  }

  $options = delete_undef_values({
    'Addr'                   => $addr,
    'InternalAddr'           => $internal_addr,
    'AutoprofBucket'         => $autoprof_bucket,
    'Datacenter'             => $datacenter,
    'Environment'            => $env,
    'NodeName'               => $node_name,
    'PIDFile'                => $pidfile,
    'TLSAddr'                => $tls_addr,
    'TLSCerts'               => $tls_certs,
    'VitreousURL'            => $vitreous_url,
    'OptionalProcessWrapper' => $optional_process_wrapper,
  })

  file { 'nydus_configuration':
    ensure  => file,
    path    => "${conf_dir}/nydus.conf.json",
    owner   => $user,
    group   => $user,
    content => template("${module_name}/nydus.conf.json.erb"),
    require => File['nydus_dir'],
  }

  # advertise service
  consul::service { 'nydus':
    ensure         => $ensure,
    port           => $port,
    consul_tags    => $consul_tags,
    check_script   => $consul_check_script,
    check_http     => "http://localhost:${port}/health",
    check_interval => '5s',
  }

  twitch_systemd::unit_file { 'nydus.service':
    ensure    => $ensure,
    content   => template("${module_name}/nydus.service.erb"),
    subscribe => File['nydus_configuration'],
  }

  $systemd_enabled = $ensure ? {
    'absent'  => false,
    'present' => true,
    default   => false,
  }

  $systemd_ensure = $systemd_enabled ? {
    true    => 'running',
    false   => 'stopped',
    default => 'stopped',
  }

  service { 'nydus':
    ensure     => $systemd_ensure,
    enable     => $systemd_enabled,
    hasrestart => true,
    restart    => 'systemctl reload nydus',
    require    => Twitch_systemd::Unit_file['nydus.service'],
    subscribe  => Twitch_systemd::Unit_file['nydus.service'],
  }

  # we require numactl to be installed. Use ensure_packages helper for pulling in
  # packages that may be declared in multiple modules
  ensure_packages(['numactl'])
  Package['numactl'] -> Service['nydus']

  $log_dir = '/var/log/nydus'
  file { $log_dir:
    ensure  => directory,
    owner   => 'syslog',
    group   => 'adm',
    mode    => '0755',
    require => Class['::rsyslog']
  }
  -> ::rsyslog::snippet { '40-nydus':
    ensure  => $ensure,
    content => template("${module_name}/rsyslog.conf.erb")
  }
  -> logrotate::rule { 'nydus':
    ensure          => $ensure,
    path            => "${log_dir}/*.log",
    missingok       => true,
    rotate          => 24,
    rotate_every    => 'hour',
    sharedscripts   => true,
    compress        => true,
    delaycompress   => false,
    compresscmd     => '/bin/bash',
    compressoptions => '-c "dd iflag=direct bs=128k status=none | gzip -1 | dd bs=128k oflag=direct status=none"',
    compressext     => '.gz',
    create          => true,
    create_mode     => '0644',
    create_owner    => 'syslog',
    create_group    => 'adm',
    postrotate      => '/usr/local/bin/rsyslog_reopen_logs.sh'
  }

}
