define twitch_pgbouncer::instance (
  $listen_port,
  $databases,
  $admin_users,
  $database_name,
  $database_env,
  $manage_service            = true,
  $enable                    = true,
  $listen_addr               = '*',
  $pool_mode                 = 'transaction',
  $max_client_conn           = '15000',
  $default_pool_size         = '20',
  $userlist                  = {},
  $ignore_startup_parameters = ['extra_float_digits','application_name'],

  # options will be automatically expanded to key = value on a new line in the ini file. This allows
  # special configuration such as 'server_reset_query' => '' to stop sending the default "discard all".
  $options                   = {},

  $rebouncer_port            = $listen_port + 1000,
  $rebouncer_hotspares       = undef,
) {

  include twitch_pgbouncer
  $config_dir = $twitch_pgbouncer::config_dir
  $socket_dir = $twitch_pgbouncer::socket_dir
  $user       = $twitch_pgbouncer::user
  $group      = $twitch_pgbouncer::group

  validate_bool($enable)
  validate_re($pool_mode, '^transaction|session$')
  validate_hash($databases, $userlist)

  $init_name     = "pgbouncer_${name}"
  $init_script   = "/etc/init.d/${init_name}"
  $default_file  = "/etc/default/${init_name}"
  $config_file   = "${config_dir}/${name}.ini"
  $userlist_file = "${config_dir}/${name}_userlist.txt"
  $hiera_users   = hiera_hash(twitch_pgbouncer::users, {})
  $users         = merge($twitch_pgbouncer::users::common_users, $userlist, $hiera_users)

  case $enable {
    true: {
      $service_ensure = 'running'
      $start_num      = '1'
    }
    false: {
      $service_ensure = 'stopped'
      $start_num      = '0'
    }
    default: {
      fail("Define twitch_pgbouncer::instance: Don't know how to handle value '${enable}' for ensure")
    }
  }

  # Init script is no longer used. A templatized systemd service unit is used.
  file { $init_script:
    ensure  => absent,
  }

  file { $default_file:
    ensure  => file,
    content => "START=${start_num}\n",
    mode    => '0644',
    owner   => 'root',
    group   => 'root',
  }

  if $rebouncer_hotspares != undef {
    twitch_pgbouncer::rebouncer { "rebouncer_${name}":
      pgbouncer_name            => $name,
      database_name             => $database_name,
      deploy_env                => $database_env,
      config_dir                => $config_dir,
      config_file               => $config_file,
      databases                 => $databases,
      hotspares                 => $rebouncer_hotspares,
      admin_users               => $admin_users,
      userlist_file             => $userlist_file,
      default_pool_size         => $default_pool_size,
      ignore_startup_parameters => $ignore_startup_parameters,
      pgbouncer_port            => $listen_port,
      rebouncer_port            => $rebouncer_port,
      max_client_conn           => $max_client_conn,
      init_name                 => $init_name,
      pool_mode                 => $pool_mode,
      socket_dir                => $socket_dir,
      pgbouncer_options         => $options,
      user                      => $user,
      group                     => $group,
    }

    if $::twitch_team == 'video' {
      twitch_nrdp::check { 'pgrebouncer':
        command  => '/usr/local/monitor_scripts/check_rebouncer.sh http://localhost:7543/nagios',
        interval => 1,
      }
    }

  } else {
    file { $config_file:
      content => template("${module_name}/pgbouncer.ini.erb"),
      mode    => '0640',
      owner   => $user,
      group   => $group,
      before  => Service["pgbouncer@${name}"]
    }
  }

  twitch_sandstorm_agent::template { "pgbouncer_${name}_userlist.txt":
    ensure          => present,
    destination     => $userlist_file,
    contents        => template("${module_name}/userlist.txt.erb"),
    restart_command => "/bin/systemctl try-reload-or-restart pgbouncer@${name}",
    mode            => '0640',
    owner           => $user,
    group           => $group,
  }

  service { "pgbouncer@${name}": }

  if $manage_service {
    # pgbouncer shouldn't subscribe directly to the config file if rebouncer is live
    # because rebouncer manages that file now
    $svc_subscriptions = $rebouncer_hotspares ? {
      undef   => [
        Twitch_sandstorm_agent::Template["pgbouncer_${name}_userlist.txt"],
        File[$config_file, $default_file],
        Twitch_systemd::Unit_file['pgbouncer@.service'],
      ],
      default => [Twitch_sandstorm_agent::Template["pgbouncer_${name}_userlist.txt"]],
    }

    Service["pgbouncer@${name}"] {
      enable    => $enable,
      ensure    => $service_ensure,
      restart   => "/bin/systemctl reload pgbouncer@${name}",
      subscribe => $svc_subscriptions,
    }
  }
}
