# Twitch Ping Service
#
# Because we preface everything with Twitch, and it is a service that pings
# itself. This acts a bit like smokeping, but better reflects actual traffic.
# It then logs this data to graphite to allow us to map and graph network
# information.
class twitch_ping_service (
  $publish_pingset   = $twitch_ping_service::params::publish_pingset,
  $seek_pingset      = $twitch_ping_service::params::seek_pingset,
  $advertise_pingset = $twitch_ping_service::params::advertise_pingset,
  $send_ports        = $twitch_ping_service::params::send_ports,
  $listen_port       = $twitch_ping_service::params::listen_port,
  $service_env       = $twitch_ping_service::params::service_env,
) inherits twitch_ping_service::params {
  validate_array($send_ports)
  if count($send_ports) < 1 {
    fail('twitch_ping_service::send_ports must contain at least one port')
  }

  if !$::ec2_placement_availability_zone {
    # baremetal pop

    # da0N --> r01, fixes a problem in some newer POPs where a switch name populates the rack_name fact
    $fixed_rack_name_1 = regsubst($::rack_name, 'da0[0-9]', 'r01')

    # Rack names should finish in the name of the POP. We already have this at the front
    # of our identity, so strip it off
    $fixed_rack_name_2 = regsubst($fixed_rack_name_1, "\\.${::pop}$", '')

    # replace any remaining '.'s with an underscore
    $fixed_rack_name_3 = regsubst($fixed_rack_name_2, '\\.', '_', 'G')

    $identity = "${::pop}.${fixed_rack_name_3}.${::hostname}"
    $bucket = $::pop
  } else {

    $fixed_hostname = $::hostname    # XXX changelog-slave-2-88a5c54d is missing ".prod"

    # Get the last character, so "us-west-2a" becomes "a". Oh, Puppet.
    $az = regsubst($::ec2_placement_availability_zone, '^(.*)(.)$', '\2')

    $identity = "${::pop}.aws${az}.${fixed_hostname}"
    $bucket = $::pop
  }

  courier::install { 'twitch-ping-service':
    env     => $service_env,
    repo    => 'video/twitch-ping-service',
    basedir => '/opt/twitch',
    require => [
      Consul::Service['twitch_ping_service'],
      Class['consul'],
    ],
  }

  consul::service { 'twitch_ping_service':
    service_name => 'twitch_ping_service',
    port         => $listen_port,
    consul_tags  => ["pingset-${advertise_pingset}", "identity=${identity}", "bucket=${bucket}", $service_env],
  }

  file { '/opt/twitch/twitch-ping-service/etc/twitch_ping_service.conf.tmpl':
    owner   => 'jtv',
    group   => 'jtv',
    mode    => '0644',
    content => template("${module_name}/twitch_ping_service.conf.tmpl.erb"),
    require => Courier::Install['twitch-ping-service'],
  }

  file { '/opt/twitch/twitch-ping-service/etc/twitch_ping_service.hcl':
    owner   => 'jtv',
    group   => 'jtv',
    mode    => '0644',
    content => template("${module_name}/twitch_ping_service.hcl.erb"),
    require => Courier::Install['twitch-ping-service'],
  }

  consul_template::service{ 'twitch_ping_service':
    source      => '/opt/twitch/twitch-ping-service/etc/twitch_ping_service.conf.tmpl',
    destination => '/opt/twitch/twitch-ping-service/etc/twitch_ping_service.conf',
    command     => 'sudo svc -h /etc/service/twitch_ping_service',
  }

  $user = 'jtv'
  $systemd_wd = '/opt/twitch/twitch-ping-service'
  $systemd_exec_cmd = '/opt/twitch/twitch-ping-service/current/twitch-ping-service /opt/twitch/twitch-ping-service/etc/twitch_ping_service.conf'

  twitch_systemd::unit_file { 'video-twitch-ping-service.service':
    ensure  => $systemd_ensure,
    content => template('twitch_ping_service/twitch_ping_service.service.erb'),
  }

  service { 'video-twitch-ping-service':
    ensure    => 'running',
    enable    => true,
    subscribe => Twitch_systemd::Unit_file['video-twitch-ping-service.service'],
    require   => [
      Courier::Install['twitch-ping-service'],
      Consul_template::Service['twitch_ping_service'],
    ]
  }

}
