class twitch_postgresql::instance::common(
  $client_hosts = undef,
  $database_name = undef,
  $enable_idle_transaction_killer = true,
  $db_environment,
  $master_db = undef,
  $parent_migrations_database_name = undef,
  $postgres_users = false,
  $puppetize_database = false,
  $replication_hosts = undef,
  $replication_targets = undef,
  $restore_wal_e = true,
  $stats_cluster = undef,
  $stats_global_interval = 10,
  $stats_lag_interval = 10,
  $stats_stat_interval = 10,
  $repmgr_topology = undef,
) inherits twitch_postgresql {
  validate_bool($restore_wal_e)

  include twitch::packages

  #Adds awscli to all boxes.
  include twitch_python::pip::awscli

  $basedir = '/var/lib/postgresql'
  $version = $postgresql::params::version
  $confdir = "/etc/postgresql/${version}/main"
  $is_master = $master_db ? {
    $::fqdn     => true,
    '127.0.0.1' => true,
    default     => false,
  }

  if $is_master and $database_name {
    $migrations_database_name = $parent_migrations_database_name ? {
      undef   => $database_name,
      default => $parent_migrations_database_name,
    }
    class { 'twitch_postgresql::lib::migrations':
      database_name => $migrations_database_name,
    }

    # XXX AGB: These should probably be done through create_resources
    # if we ever have more than just 2. 2015-05-07
    consul::service { "migrations-${database_name}":
      consul_tags => [$db_environment],
    }
    if $parent_migrations_database_name {
      consul::service { "migrations-${parent_migrations_database_name}":
        consul_tags => [$db_environment],
      }
    }
  }


  # Same cron as found in manifests/classes/cron/pg_log_cleanup.pp
  cron { 'pg_log_cleanup':
    command => 'find /var/log/postgresql/ -name "postgresql*" -type f -mtime +7 -delete',
    require => Package['postgresql-server'],
    user    => 'postgres',
    hour    => '0',
  }

  include twitch_postgresql::lib::bindir

  postgresql::server::config_entry { 'hot_standby': value => 'on'}
  postgresql::server::config_entry { 'log_destination': value => 'csvlog'}
  postgresql::server::config_entry { 'log_directory': value => '/var/log/postgresql'}
  postgresql::server::config_entry { 'wal_level': value => 'logical'}
  postgresql::server::config_entry { 'logging_collector': value => 'on' }
  postgresql::server::config_entry { 'stats_temp_directory': value => '/dev/shm/pg_stat_tmp' }

  $entries = hiera_hash("${module_name}::config_entries")
  create_resources(postgresql::server::config_entry, $entries)

  postgresql::server::pg_hba_rule { 'local nagios connections':
    description => 'allow nagios to connect locally',
    type        => 'local',
    database    => 'all',
    user        => 'nagios',
    auth_method => 'peer',
    order       => '070',
  }

  postgresql::server::pg_hba_rule { 'local replication':
    description => 'allow replication user access for pg_basebackup',
    type        => 'local',
    database    => 'replication',
    user        => 'replication',
    auth_method => 'peer',
    order       => '040',
  }
  $replication_auth = str2bool($::clean) ? {
    true  => 'md5',
    default => 'trust',
  }
  $replication_defaults = {
    user => 'replication',
    auth_method => $replication_auth,
  }

  if $replication_hosts != undef {
    create_resources(twitch_postgresql::replica_pg_hba, $replication_hosts, $replication_defaults)

    if $repmgr_topology and $database_name {
      $repmgr_repmgr_defaults = {
        database => 'repmgr',
      }
      $repmgr_replication_defaults = {
        database => 'replication',
      }

      $repmgr_repmgr_hosts      = prefix($replication_hosts, 'repmgr-')
      $repmgr_replication_hosts = prefix($replication_hosts, 'replication-')
      create_resources(postgresql_repmgr::lib::hba_rule, $repmgr_repmgr_hosts, $repmgr_repmgr_defaults)
      create_resources(postgresql_repmgr::lib::hba_rule, $repmgr_replication_hosts, $repmgr_replication_defaults)
    }
  }
  if $replication_targets != undef {
    create_resources(twitch_postgresql::replication_target, $replication_targets, $replication_defaults)
  }

  if $client_hosts != undef {
    $dbname = $database_name ? {
      undef   => 'all',
      default => $database_name,
    }
    $client_defaults = {
      database    => $dbname,
      user        => 'all',
      auth_method => 'md5',
    }
    create_resources(twitch_postgresql::client_pg_hba, $client_hosts, $client_defaults)
  }

  $is_aws = str2bool(hiera('aws', false))
  $ensure_restore_wal_e = $restore_wal_e ? {
    true  => 'present',
    false => 'absent',
  }
  file { "${basedir}/bin/restore-wal-e":
    ensure  => $ensure_restore_wal_e,
    mode    => '0755', # -rwxr-xr-x
    owner   => 'postgres',
    group   => 'postgres',
    require => File["${basedir}/bin"],
    content => template("${module_name}/restore-wal-e.erb"),
  }

  # for debug output on the puppet client
  notify {"Repmgr Topology: ${repmgr_topology} for ${::fqdn} / ${database_name}":}

  if $repmgr_topology and $database_name {
    class { 'postgresql_repmgr':
      db_cluster_name   => $database_name,
      db_env            => $db_environment,
      rep_node_id       => $repmgr_topology[$::fqdn]["id"],
      rep_node_name     => $repmgr_topology[$::fqdn]["name"],
      upstream_id       => $repmgr_topology[$::fqdn]["upstream"],
      failover_priority => $repmgr_topology[$::fqdn]["priority"],
      is_master         => $is_master,
    }
  }
}
