# sets up puppet4 agent
class twitch_puppet4::agent(
  $puppet_server       = $twitch_puppet4::params::puppet_server,
  $puppet_ca_server    = $twitch_puppet4::params::puppet_ca_server,
  $puppet_certname     = $twitch_puppet4::params::puppet_certname,
  $puppet_run_style    = 'manual',
  $puppet_run_command  = '/opt/puppetlabs/bin/puppet agent --no-daemonize --onetime --logdest syslog > /dev/null 2>&1',
  $puppet_version      = $twitch_puppet4::params::puppet_version,
  $environment         = 'production',
  $splay               = false,
  $puppet_conf         = $twitch_puppet4::params::puppet_conf,
  $puppet_logdir       = $twitch_puppet4::params::puppet_logdir,
) inherits twitch_puppet4::params {
  # Setting this to anything besides 30 will not do what you want.
  $puppet_run_interval = 30

    package { 'puppet-agent':
      ensure => installed,
    }

    package { 'libuser':
      ensure => present,
    }

    group { $twitch_puppet4::params::puppet_group:
      ensure => 'present',
      system => true,
    }

    user { $twitch_puppet4::params::puppet_user:
      ensure => 'present',
      system => true,
      gid    => $twitch_puppet4::params::puppet_group,
    }

    file { $twitch_puppet4::params::puppet_confdir:
      ensure => 'directory',
      owner  => $twitch_puppet4::params::puppet_user,
      group  => $twitch_puppet4::params::puppet_group,
      mode   => '0755',
    }

    file { $puppet_conf:
      ensure => 'file',
      owner  => $twitch_puppet4::params::puppet_user,
      group  => $twitch_puppet4::params::puppet_group,
      mode   => '0644',
    }
    # ensure that puppet is not running and will not start up on boot
    $service_ensure = 'stopped'
    $service_enable = false

    # Run puppet as a cron - this saves memory and avoids the whole problem
    # where puppet locks up for no reason. Also spreads out the run intervals
    # more uniformly.
    $time1  =  fqdn_rand($puppet_run_interval, 'puppet-cron')
    $time2  =  $time1 + 30
    if $facts['vagrant'] == undef {
      twitch_cron { 'puppet-client':
        command     => $puppet_run_command,
        user        => 'root',
        # run twice an hour, at a random minute in order not to collectively stress the puppetmaster
        environment => 'PATH=/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin',
        minute      => "${time1},${time2}",
      }
    }

    file { '/etc/default/puppet':
      mode    => '0644',
      owner   => 'root',
      group   => 'root',
      require => Package['puppet-agent'],
      content => template('twitch_puppet/puppet.default.erb'),
    }

    service { 'puppet':
      ensure     => $service_ensure,
      enable     => $service_enable,
      hasstatus  => true,
      hasrestart => true,
      require    => Package['puppet-agent'],
    }

    # override local ini_setting resources
    Ini_setting {
      path    => $puppet_conf,
      section => 'agent',
      require => File[$puppet_conf],
      before  => Package['puppet-agent'],
    }

    ini_setting { 'puppet_ca_server':
      section => 'main',
      setting => 'ca_server',
      value   => $puppet_ca_server,
    }

    ini_setting {'puppet_stringify_facts':
      ensure  => present,
      section => 'main',
      setting => 'stringify_facts',
      value   => false,
    }

    ini_setting { 'puppet_logdir':
      section => 'main',
      setting => 'logdir',
      value   => $puppet_logdir,
    }

    # needs to be set in main
    ini_setting { 'puppetagent_ca_server':
      ensure  => 'absent',
      setting => 'ca_server',
    }

    if $puppet_certname {
      ini_setting { 'puppetagent_certname':
        setting => 'certname',
        value   => $puppet_certname,
      }
    }

    ini_setting { 'puppetagent_server':
      setting => 'server',
      value   => $puppet_server,
    }

    ini_setting { 'puppetagent_environment':
      setting => 'environment',
      value   => $environment,
    }

    ini_setting { 'puppetagent_runinterval':
      setting => 'runinterval',
      value   => $puppet_run_interval * 60,
    }

    ini_setting { 'puppetagent_splay':
      setting => 'splay',
      value   => $splay
    }
}
