# sets up puppet agent
class twitch_puppet::agent::service (
  $puppet_run_style    = $twitch_puppet::params::puppet_run_style,
  $puppet_run_interval = $twitch_puppet::params::puppet_run_interval,
  $puppet_run_command  = $twitch_puppet::params::puppet_run_command,
) inherits twitch_puppet::params {
  include twitch_puppet::agent

  # Validation
  validate_string($puppet_run_style, $puppet_run_command)
  validate_integer($puppet_run_interval)

  case $puppet_run_style {

    'service': {
      $service_ensure = 'running'
      $service_enable = true

      systemd::service::dropin { 'puppet':
        environment_files => ['-/etc/profile.d/proxy.sh'],
      }
    }
    'cron','daily_core': {
      # ensure that puppet is not running and will start up on boot
      $service_ensure = 'stopped'
      $service_enable = false

      # Run puppet as a cron - this saves memory and avoids the whole problem
      # where puppet locks up for no reason. Also spreads out the run intervals
      # more uniformly. Runs 2x per hour when set to 30 (should be)
      $time1  =  fqdn_rand($puppet_run_interval, 'puppet-cron')
      $time2  =  ($time1 + $puppet_run_interval) % 60

      # Only apply core if this is set.
      if  $puppet_run_style == 'daily_core' {
        $cron_command = "${puppet_run_command} --tags core 2>&1 | logger -p local3.info -t puppet-cron"
        $cron_hour    = fqdn_rand(24, 'puppet-cron-hour')
        $cron_minute  = fqdn_rand(60, 'puppet-cron-minute')
      }
      else {
        $cron_command = join([
            "/usr/bin/pkill -xf '/usr/bin/ruby /usr(/local|)/bin/puppet agent' ;",
            $puppet_run_command, '2>&1 | logger -p local3.info -t puppet-cron',
        ], ' ')
        $cron_minute  = "${time1},${time2}"
        $cron_hour    = '*'
      }

      if $facts['vagrant'] == undef {
        twitch_cron { 'puppet-client':
          command     => $cron_command,
          user        => 'root',
          # run twice an hour, at a random minute in order not to collectively stress the puppetmaster
          environment => 'PATH=/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin',
          minute      => $cron_minute,
          hour        => $cron_hour,
        }
      }
      # on boot puppet run on the netbooted machines is done in the provisioner_checkin.sh script
      # please check twitch_netboot module
      if $facts['vagrant'] == undef and $::cmdline['boot'] != 'live' or $::packer_build_name) {
        # Run puppet on boot too.
        twitch_cron { 'puppet-agent-on-boot':
          ensure      => present,
          command     => $cron_command,
          user        => 'root',
          environment => 'PATH=/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin',
          minute      => '@reboot',
          hour        => '',
          monthday    => '',
          month       => '',
          weekday     => '',
        }
      }
    }
    'manual': {
      # This wont remove the installed crontab if you flip from cron->manual.
      $service_ensure = false
      $service_enable = false
    }
    default: {
      fail('Unsupported puppet run style in Class[\'twitch_puppet::agent\']')
    }
  }

  file { '/etc/default/puppet':
    mode    => '0644',
    owner   => 'root',
    group   => 'root',
    content => template('twitch_puppet/puppet.default.erb'),
  }

  service { 'puppet':
    ensure     => $service_ensure,
    enable     => $service_enable,
    hasstatus  => true,
    hasrestart => true,
  }

}
