# Moving the master config items out of the master.pp class into its own
class twitch_puppet::master::config (
    $puppet_user            = $twitch_puppet::params::puppet_user,
    $puppet_group           = $twitch_puppet::params::puppet_group,
    $puppet_vardir          = $twitch_puppet::params::puppet_vardir,
    $puppet_confdir         = $twitch_puppet::params::puppet_confdir,
    $puppet_ssldir          = $twitch_puppet::params::puppet_ssldir,
    $puppet_environment_dir = $twitch_puppet::params::puppet_environment_dir,
    $puppet_production_link = $twitch_puppet::params::puppet_production_link,
    $puppet_conf            = $twitch_puppet::params::puppet_conf,
    $puppet_certname        = $twitch_puppet::params::puppet_certname,
    $puppet_pidfile         = $twitch_puppet::params::puppet_pidfile,
    $puppet_rackdir         = $twitch_puppet::params::puppet_rackdir,
    $puppet_unicorn_pidfile = $twitch_puppet::params::puppet_unicorn_pidfile,
    $puppet_unicorn_socket  = $twitch_puppet::params::puppet_unicorn_socket,
    $git_server             = $twitch_puppet::params::git_server,
    $git_project            = twitch_puppet::params::git_project,
    $puppet_server          = $twitch_puppet::params::puppet_server,
    $puppet_ca_server       = $twitch_puppet::params::puppet_ca_server,
    $puppet_logdir          = $twitch_puppet::params::puppet_logdir,
    $dns_alt_names          = ["video-puppet.${::domain}", $puppet_server],
    $environmentpath        = '$confdir/environments',
    $modulepath             = '$confdir/environments/$environment/modules',
    $manifest               = '$confdir/environments/$environment/manifests/site.pp',
    $templatedir            = '$confdir/environments/$environment/templates',
    $hiera_config           = '$confdir/hiera.yaml',
    $reports                = 'none',
    $environments           = 'directory',
    $reporturl              = undef,
    $external_nodes         = undef,
    $node_terminus          = undef,
    $always_cache_features  = false,
    $strict_variables       = false,
    $trusted_node_data      = true,
    $ca                     = false,
    $bootstrap              = false,
) inherits twitch_puppet::params {
  include twitch_puppet::master

  # Validate all the things
  validate_bool($always_cache_features, $strict_variables, $trusted_node_data, $ca, $bootstrap)
  validate_string($environments, $puppet_user, $puppet_group, $puppet_conf, $modulepath, $manifest, $puppet_ca_server)
  validate_string($templatedir, $hiera_config, $reports, $reporturl, $external_nodes, $node_terminus, $puppet_certname)
  validate_absolute_path($puppet_vardir, $puppet_confdir, $puppet_ssldir, $puppet_environment_dir, $puppet_logdir)
  validate_absolute_path($puppet_conf, $puppet_pidfile, $puppet_rackdir, $puppet_unicorn_pidfile, $puppet_unicorn_socket)
  validate_array($dns_alt_names)

  case $environments {
    'config': {
      $setting_config    = 'present'
      $setting_directory = 'absent'
    }
    'directory': {
      $setting_config    = 'absent'
      $setting_directory = 'present'
    }
    default: { fail("Unknown value for \$environments ${environments}") }
  }

  $dns_alt_names_string = join($dns_alt_names, ',')

  # Calculate here instead of in multiple erb
  $worker_count = $::processorcount * 2

  # Get worker max memory, leaving 1024 for other things
  $worker_max_memory = ($::memorysize_mb - 1024) / $worker_count

  # Workers will be killed between min and max with a rand
  $worker_min_memory = $worker_max_memory - 100

  file { $puppet_vardir:
    ensure => 'directory',
    owner  => $puppet_user,
    group  => $puppet_group,
    mode   => '0755',
  }

  file { $puppet_ssldir:
    ensure  => 'directory',
    owner   => $puppet_user,
    group   => $puppet_group,
    mode    => '0771',
    require => File[$puppet_vardir],
  }

  file { "${puppet_confdir}/auth.conf":
    owner   => $puppet_user,
    group   => $puppet_group,
    mode    => '0644',
    source  => "puppet:///modules/${module_name}/master/auth.conf",
    require => File[$puppet_confdir],
  }

  file { $puppet_environment_dir:
    ensure => directory,
    owner  => $puppet_user,
    group  => $puppet_group,
    mode   => '0755',
  }

  file { "${puppet_environment_dir}/production":
    ensure => directory,
    owner  => $puppet_user,
    group  => $puppet_group,
    mode   => '0755',
  }

  # unicorn/rack related
  file { $puppet_rackdir:
    ensure  => 'directory',
    owner   => $puppet_user,
    group   => $puppet_group,
    mode    => '0755',
    require => File[$puppet_confdir],
  }

  file { "${puppet_rackdir}/config.ru":
    ensure  => 'present',
    owner   => $puppet_user,
    group   => $puppet_group,
    mode    => '0644',
    content => template('twitch_puppet/config.ru.erb'),
    require => File[$puppet_rackdir],
  }

  file { "${puppet_rackdir}/unicorn.rb":
    ensure  => 'present',
    owner   => $puppet_user,
    group   => $puppet_group,
    mode    => '0644',
    content => template('twitch_puppet/unicorn.rb.erb'),
    notify  => Class['twitch_puppet::master::monitoring'],
    require => File[$puppet_rackdir],
  }

  # Lets manage the top-level hiera.yaml outside any environment,
  # since puppet.conf only refs one before $env is parsed
  file { "${puppet_confdir}/hiera.yaml":
    ensure  => file,
    path    => "${puppet_confdir}/hiera.yaml",
    source  => 'puppet:///modules/twitch_puppet/hiera.yaml',
    owner   => $puppet_user,
    group   => $puppet_group,
    mode    => '0644',
    require => File[$puppet_confdir],
  }

  Ini_setting {
    path    => $puppet_conf,
    section => 'master',
    require => File[$puppet_conf],
  }

  ini_setting {'puppetmaster_ssl_client_header':
    ensure  => absent,
    section => 'master',
    setting => 'ssl_client_header',
  }

  ini_setting {'puppetmaster_ssl_client_verify_header':
    ensure  => absent,
    section => 'master',
    setting => 'ssl_client_verify_header,'
  }

  ini_setting { 'puppetmaster_ssldir':
    section => 'main',
    setting => 'ssldir',
    value   => $puppet_ssldir,
  }

  ini_setting { 'puppetmaster_dns_alt_names':
    section => 'main',
    setting => 'dns_alt_names',
    value   => $dns_alt_names_string,
  }

  ini_setting { 'puppetmaster_modulepath':
    ensure  => $setting_config,
    setting => 'modulepath',
    value   => $modulepath,
  }

  ini_setting { 'puppetmaster_manifest':
    ensure  => $setting_config,
    setting => 'manifest',
    value   => $manifest,
  }

  ini_setting {'puppetmaster_environmentpath':
    ensure  => $setting_directory,
    section => 'main',
    setting => 'environmentpath',
    value   => $environmentpath,
  }

  if $external_nodes != undef {
    ini_setting {'puppetmaster_encconfig':
      setting => 'external_nodes',
      value   => $external_nodes,
    }

    ini_setting {'puppetmaster_nodeterminus':
      setting => 'node_terminus',
      value   => 'exec',
    }
  }
  elsif $node_terminus != undef {
    ini_setting {'puppetmaster_nodeterminus':
      setting => 'node_terminus',
      value   => $node_terminus,
    }
  }

  ini_setting {'puppetmaster_hieraconfig':
    setting => 'hiera_config',
    value   => $hiera_config,
  }

  ini_setting { 'puppetmaster_ca':
    setting => 'ca',
    value   => $ca,
  }

  ini_setting { 'puppetmaster_templatedir':
    ensure  => $setting_config,
    setting => 'templatedir',
    value   => $templatedir,
  }

  ini_setting { 'puppetmaster_cache_features':
    setting => 'always_cache_features',
    value   => $always_cache_features,
  }

  ini_setting { 'puppetmaster_strict_variables':
    setting => 'strict_variables',
    value   => $strict_variables,
  }

  ini_setting { 'puppetmaster_trusted_node_data':
    setting => 'trusted_node_data',
    value   => $trusted_node_data,
  }

  $reports_ensure = $reporturl ? {
    default => 'present',
    undef   => 'absent',
  }

  ini_setting { 'puppetmaster_reporturl':
    ensure  => $reports_ensure,
    setting => 'reporturl',
    value   => $reporturl,
  }

  ini_setting { 'puppet_config_version':
    ensure  => $setting_config,
    section => 'production',
    setting => 'config_version',
    value   => '$confdir/environments/production/version.sh',
  }

  # this is hardcoded in upstream deb package
  # it needs to be set correctly
  # to stop the service that is autostarted after package install
  ini_setting { 'puppetmaster_pidfile':
    setting => 'pidfile',
    value   => $puppet_pidfile,
  }

  ini_setting { 'puppetmaster_reports':
    setting => 'reports',
    value   => $reports,
  }

  # Removes report files/directories after 1 week
  tidy { '/var/lib/puppet/reports':
    age     => '1w',
    recurse => true,
  }

  # Ensure removal of [production] section in puppet since ini_setting does
  # not support it
  file_line { 'puppet_ini_production_section':
    ensure  => $setting_config,
    path    => $puppet_conf,
    line    => '[production]',
    require => Ini_setting['puppet_config_version'],
  }

  ## puppet master bootstrap and certiticate generation
  if ! $ca and $bootstrap {
    # after the puppetmaster server is installed, if we're not the 1 CA
    # and we dont already have a certificate in the certs dir we will
    # automatically generate a CSR with dns_alt_names (from config),
    # and send that request to our current CA, based on config
    $puppet_crt_cmd = 'puppet certificate --ca-location=remote'
    $master_bootstrapper = "${puppet_vardir}/master_bootstrapper.sh"

    # The master bootstrapper includes a part that will fork into the background
    # and wait until the cert request is approved (manually) on the CA server,
    # then will restart services that fail to start without a proper certificate.
    file { $master_bootstrapper:
      content => template("${module_name}/master_bootstrapper.sh.erb"),
      owner   => 'root',
      group   => 'root',
      mode    => '0700',
      require => File[$puppet_vardir],
    }

    exec { 'bootstrap puppetmaster':
      command => $master_bootstrapper,
      creates => "${puppet_vardir}/master_bootstrapped",
      require => [
        File[$master_bootstrapper],
        Ini_setting['puppetmaster_dns_alt_names'],
      ],
    }

    # Do initial repo checkout so the github webhook receiver works
    vcsrepo { "${puppet_environment_dir}/master":
      ensure   => 'present',
      provider => 'git',
      source   => "git@${git_server}:${git_project}.git",
      user     => $puppet_user,
      require  => Twitch_Sandstorm_Agent::Template['webhook-ssh-key-priv'],
    }
  }
}
