"""
Config-related useful stuff
"""

from logging import ERROR, WARNING, INFO, DEBUG, getLogger, StreamHandler
from optparse import OptionGroup
import inspect
import os
import sys


# global logger for this library
log = getLogger('ttv.config')  # pylint: disable=C0103
#log.addHandler(StreamHandler())


#############################################################################
#
# config.var
#

def var(name, default, namespace=None, vartype=None):
    """Declare config envariables like this:

    config.var('VARNAME', 12)

    or

    config.var('VARNAME', 12, namespace='ttv.foo', vartype=int)

    default_value can be overridden by VARNAME envariable:

    env VARNAME=14 ./foo.py

    namespace is inferred if not provided.

    vartype defaults to type(default_value). It may also be specified,
    as an actual type or as the (quoted) name of a type, although the
    latter is deprecated.
    """
    # pylint: disable=W0122

    # grab caller's namespace (don't try this at home)
    if namespace is None:
        namespace = inspect.stack()[1][0].f_globals['__name__']

    # infer vartype if it's not specified
    if vartype is None:
        vartype = type(default)

    if type(vartype) == type(type):
        setattr(sys.modules[namespace], name, vartype(os.environ.get(name, default)))
    else:
        # deprecated
        exec("""sys.modules['%s'].%s = %s(os.environ.get('%s', '%s'))""" %
             (namespace, name, vartype, name, default))


#############################################################################
#
# logging goo
#

# add all loggers here...irrelevant ones don't hurt

LOGGERS = {'normal': ('', 'ttv', 'ttv_alertlogger', 'ttv.archive', 'ttv.config', 'ttv.db', 'ttv.db.netdb',
                      'ttv.db.sitedb', 'ttv.depoloy', 'ttv.kimchi', 'ttv.nagios', 'ttv.s3', 'ttv.usher'),
           'noisy': (),
           'vnoisy': ('tftpy', 'requests.packages.urllib3.connectionpool',),}

LOG_LEVELS = {
    +3 : {'normal': DEBUG,   'noisy': DEBUG,    'vnoisy': DEBUG,},
    +2 : {'normal': DEBUG,   'noisy': DEBUG,    'vnoisy': INFO,},
    +1 : {'normal': DEBUG,   'noisy': INFO,     'vnoisy': WARNING},
     0 : {'normal': INFO,    'noisy': WARNING,  'vnoisy': WARNING},
    -1 : {'normal': WARNING, 'noisy': WARNING,  'vnoisy': WARNING},
    -2 : {'normal': ERROR,   'noisy': ERROR,    'vnoisy': ERROR},
}

def set_log_level(log_level):
    """Set logging noisiness. + is noisier, - is quieter."""
    if log_level > max(LOG_LEVELS.keys()):
        log_level = max(LOG_LEVELS.keys())
    elif log_level < min(LOG_LEVELS.keys()):
        log_level = min(LOG_LEVELS.keys())

    for noisiness, loggers in LOGGERS.items():
        for logger in loggers:
            getLogger(logger).setLevel(LOG_LEVELS[log_level][noisiness])


def add_verbosity_options(parser):
    """An OptionParser OptionGroup for verbosity and quietness"""

    verbosity = OptionGroup(parser, 'Verbosity options',
                            '')
    verbosity.add_option('-v', '--verbose', dest='verbose', action='count',
                         default=0,
                         help='verbose (repeat for increased verbosity)')
    verbosity.add_option('-q', '--quiet', dest='quiet', action='count',
                         default=0,
                         help='quiet (repeat for increased quietness)')
    parser.add_option_group(verbosity)
