"""
Generic db-related library functions
"""

# pylint: disable=C0111,W0201,E0602
# C0111: missing-docstring
# W0201: attribute-defined-outside-init
# E0602: undefined-variable

import psycopg2
import sqlite3
from logging import getLogger, StreamHandler
from time import sleep, time
import sys
from inspect import currentframe, getouterframes

import ttv
from ttv import config
from ttv import borg


# global logger for this library
log = getLogger('ttv.db')  # pylint: disable=C0103
#log.addHandler(StreamHandler())


config.var('DB_RETRY_SECS', 1.0)
config.var('DBRW_DRYRUN', 0)


#############################################################################

class Db(object):
    """A connection to a database."""
    # pylint: disable=R0902
    # pylint:disable=R0913
    
    def __init__(self, database, host=None, user=None, port=None, password=None, engine='psycopg2', readonly=False):
        
        # pylint: disable=R0913
        self.database = database
        self.host = host
        self.user = user
        self.port = port
        self.password = password
        self.engine = engine
        self.readonly = readonly
        self.refresh()
            
    def refresh(self):
        if self.engine == 'psycopg2':
            self.conn = psycopg2.connect(database=self.database,
                                         host=self.host,
                                         user=self.user,
                                         port=self.port,
                                         password=self.password)
            try:
                self.conn.set_session(readonly=self.readonly, autocommit=True)
            except (AttributeError) as e:
                log.warning('set_session not available in this version of psycopg2')
            except (psycopg2.InterfaceError, psycopg2.OperationalError, psycopg2.DatabaseError) as e:
                log.warning('set_session failed: %s; retrying', e)
                sleep(DB_RETRY_SECS)
                # at this point, if it fails, it fails
                self.conn.set_session(readonly=self.readonly, autocommit=True)
        elif self.engine == 'sqlite3':
            self.conn = sqlite3.connect(self.database)
            
        self.cursor = self.conn.cursor()

    def execute(self, sql, sqlargs=()):
        # - sqlargs may be absent, sequence, or dictionary
        # - bind variable syntax varies from backend to backend :-(
        # - frame is supposed to be a named tuple but doesn't seem to be
        prog = sys.argv[0]
        frame = getouterframes(currentframe())[1]
        progfile, progline = frame[1], frame[2]

        sql += '  -- EngOps %s (in %s, line %s)' % (prog, progfile, progline)
        log.debug('%s %s', sql, sqlargs)
        try:
            start = time()
            self.cursor.execute(sql, sqlargs)
        except (psycopg2.InterfaceError, psycopg2.OperationalError, psycopg2.DatabaseError) as e:
            # wait one second, and retry one time. this seems to be helpful sometimes :-\
            # (both numbers are arbitrary)
            log.warning('"%s %s" failed: %s; retrying', sql, sqlargs, e)
            sleep(DB_RETRY_SECS)
            self.refresh()
            start = time()
            self.cursor.execute(sql, sqlargs)

        log.debug('query completed in %.2fs', time() - start)
        return self.cursor

    def commit(self):
        self.conn.commit()

    def close(self):
        """if you close it, you can't ever use it again"""
        self.conn.close()


#############################################################################

@borg
class DbRo(Db):
    """A read-only connection to a database. borg unless more than one
    is needed.
    """

    def __init__(self, database, host=None, user=None, port=None, password=None, engine='psycopg2'):
        # pylint: disable=R0913
        super(DbRo, self).__init__(database, host, user, port, password, engine=engine, readonly=True)


class Table(object):
    """Each row of a table becomes a tuple and a bunch of kvs. Use on
    smaller tables only, and perhaps with @objcached(). If a subclass
    wants to mess with the tuples, it should call set_tuples().

    Note, this does not track changes to the database! So it will get
    stale immediately! Use it immediately if you need the
    latest and greatest.
    """

    def __init__(self, db, table, columns):  # pylint: disable=C0103
        self.db = db  # pylint: disable=C0103
        self.table = table
        self.columns = columns
        self.refresh()

    def refresh(self):
        self.set_tuples(self.db.execute('SELECT %s FROM %s' % (','.join(self.columns), self.table)).fetchall())  # TODO

    def set_tuples(self, tuples):
        self.tuples = tuples
        self.kvs = [dict(zip(self.columns, x)) for x in self.tuples]

    def get_tuples(self):
        return self.tuples

    def get_kvs(self):
        return self.kvs

    def lookup_by_id(self, id_):
        try:
            getattr(self, '_by_id')
        except AttributeError:
            self._by_id = dict([(x['id'], x) for x in self.get_kvs()])
        return self._by_id[id_]


#############################################################################

class DbRw(Db):
    """A read-write connection to a database."""
    # pylint: disable=R0913

    def __init__(self, database, host=None, user=None, port=None, password=None, engine='psycopg2'):
        super(DbRw, self).__init__(database, host, user, port, password, engine=engine, readonly=False)


#############################################################################
