"""
Functions for getting stuff from kimchi
"""

from logging import getLogger
import requests
from functools import partial

from ttv import requests_wrapper, config, cached_http_getter, cached

log = getLogger('ttv.kimchi')  # pylint: disable=C0103


config.var('KIMCHI_SCHEME', 'https')
config.var('KIMCHI_HOST', 'kimchi-api.justin.tv')
config.var('KIMCHI_PORT', 443)
config.var('KIMCHI_URL', '%s://%s:%s' % (KIMCHI_SCHEME, KIMCHI_HOST, KIMCHI_PORT))  # pylint: disable=E0602

config.var('BRL_CACHE_TIME', 300)


def post_kimchi(objtype, **kwargs):
    url = '%s/rest/%s/' % (KIMCHI_URL, objtype)  # pylint: disable=E0602
    return requests_wrapper(requests.post, url, data=kwargs, verify=False)

post_object = partial(post_kimchi, 'objects')  # pylint: disable=C0103
post_netinterface = partial(post_kimchi, 'net_interfaces')  # pylint: disable=C0103


def get_kimchi(objtype, **kwargs):
    url = '%s/rest/%s/?' % (KIMCHI_URL, objtype)  # pylint: disable=E0602
    for k, v in kwargs.items():
        url += '%s=%s&' % (k, v)
    return requests_wrapper(requests.get, url, verify=False)

get_objects = partial(get_kimchi, 'objects')  # pylint: disable=C0103
get_domains = partial(get_kimchi, 'dns_domains')  # pylint: disable=C0103
get_statuses = partial(get_kimchi, 'object_statuss')  # pylint: disable=C0103
get_mandms = partial(get_kimchi, 'make_and_models')  # pylint: disable=C0103


# Note: stuff here gets cached for 5 minutes. Seems fine for now--can
# change later if needed.
@cached(BRL_CACHE_TIME, dont_cache_none=True, use_stale_non_none=True)  # pylint: disable=E0602
def border_router_list(cluster=None):
    """do the kimchi-api lookup, and fill in some blanks."""
    # pylint: disable=E1103
    
    r = get_objects(type='router', status='In Use', name__contains='br0')
    if r is None:
        log.warning('could not get a fresh list of border routers')
        return None

    routers = r.json
    try:
        for router in routers:
            url = router['domain']
            r = requests_wrapper(requests.get, url, verify=False)
            router['domain_name'] = domain_name = r.json['name']
            router['fqdn'] = '.'.join((router['name'], domain_name))
            if not domain_name.endswith('justin.tv'):
                log.error('domain name %s for router %s does not compute', domain_name, router['fqdn'])
                continue
            elif domain_name == 'justin.tv':
                router['cluster'] = 'sfo01'
            else:
                router['cluster'] = domain_name[:-len('.justin.tv')]
    except (KeyError, IndexError, AttributeError):
        log.error('border_router_list() failed: unexpected response %s from %s ', r.text, url)
        return None

    if cluster is not None:
        routers = [r for r in routers if r['cluster'] == cluster]
        
    return routers
