"""
s3-related
"""

# pylint: disable=C0103

import time
import hmac
import hashlib
import base64
import httplib
from logging import getLogger

from ttv import config


# global logger for this library
log = getLogger('ttv.s3')


config.var('S3HOST', 's3.amazonaws.com', namespace='ttv.s3')


def _common(method, bucket, name, ak=None, sk=None, data=None, content_type='', **more_headers):
    """Perform S3 operation"""

    # pylint: disable=R0913

    path = '/%s/%s' % (bucket, name)
    headers = {
        'Date' : time.strftime('%a, %d %b %Y %H:%M:%S GMT', time.gmtime()),
        'x-amz-acl': 'public-read',
    }
    canon = '\n'.join([
        method, 
        '',
        content_type,
        headers['Date'],
        'x-amz-acl:' + headers['x-amz-acl'],  
        path
    ])
    h = hmac.new(sk, canon, hashlib.sha1)
    sig = base64.encodestring(h.digest()).strip()
    if content_type:
        headers['Content-Type'] = content_type
    if data is not None:
        headers['Content-Length'] = len(data)
    headers['Authorization'] = 'AWS ' + ak + ':' + sig
    headers.update(**more_headers)
    
    connection = httplib.HTTPConnection(S3HOST)  # pylint: disable=E0602
    connection.request(method, path, data, headers)
    response = connection.getresponse() 
    
    log.debug('response is %s', response)

    return response


def put(bucket, name, ak=None, sk=None, data=None, content_type='', **more_headers):
    """Upload the given data to S3"""

    # pylint: disable=R0913

    return _common('PUT', bucket, name, ak=ak, sk=sk, data=data, 
                   content_type=content_type, **more_headers)


def delete(bucket, name, ak=None, sk=None, **more_headers):
    """Delete the given data from S3"""
    return _common('DELETE', bucket, name, ak=ak, sk=sk, **more_headers)


def get(bucket, name, ak=None, sk=None, **more_headers):
    """Delete the given data from S3"""
    return _common('GET', bucket, name, ak=ak, sk=sk, **more_headers)
