#!/usr/bin/env python

"""
Test ttv.db functionality.
"""

# pylint: disable=R0904,C0111,C0103
# R0904: too-many-public-methods
# C0111: missing-docstring
# C0103: invalid-name


import unittest
import sys
from os.path import dirname, abspath

sys.path.insert(0, dirname(dirname(dirname(abspath(__file__)))))
import ttv.db

class TestMisc(unittest.TestCase):

    def test_trivial(self):
        self.assertTrue(True)


def create_test_db():
    db = ttv.db.DbRw(':memory:', engine='sqlite3')
    db.execute('CREATE TABLE t(id INTEGER, c1 INTEGER, c2 INTEGER);')
    db.execute("""
INSERT INTO t 
      SELECT 1, 0, 'a' 
UNION SELECT 2, 0, 'b'
UNION SELECT 3, 1, 'a';""")
    return db


class TestParamSub(unittest.TestCase):

    def setUp(self):
        self.db = create_test_db()

    def test_trivial(self):
        self.assertEqual(self.db.execute('SELECT 1;').fetchall(), [(1,)])

    def test_no_params(self):
        self.assertEqual(self.db.execute('SELECT * FROM t WHERE c1 = 0;').fetchall(), [(1, 0, 'a'), (2, 0, 'b')])

    def test_one_params(self):
        self.assertEqual(self.db.execute('SELECT * FROM t WHERE c1 = ?', (1,)).fetchall(), [(3, 1, 'a')])

    def test_two_params(self):
        self.assertEqual(self.db.execute('SELECT * FROM t WHERE c1 = ? and c2 = ?', (1, 'b')).fetchall(), [])


class TestTable(unittest.TestCase):
    
    def setUp(self):
        self.db = create_test_db()

    def test_table(self):
        table = ttv.db.Table(self.db, 't', ('c1', 'c2'))
        self.assertEqual(table.get_tuples(), [(0, 'a'), (0, 'b'), (1, 'a')])
        self.assertEqual(table.get_kvs(), [{'c1': 0, 'c2': 'a'}, 
                                           {'c1': 0, 'c2': 'b'}, 
                                           {'c1': 1, 'c2': 'a'}])
    
    def test_lookup_by_id(self):
        table = ttv.db.Table(self.db, 't', ('id', 'c1'))
        self.assertEqual(table.lookup_by_id(3), {'id': 3, 'c1': 1})

if __name__ == '__main__':
    unittest.main()
