#!/usr/bin/env python

"""
Test ttv
"""

# pylint: disable=R0904,C0111,R0903,C0103
# R0904: too-many-public-methods
# C0111: missing-docstring
# R0903: too-few-public-methods
# C0103: invalid-name


import unittest
import sys
from os.path import dirname, abspath, isfile
import os.path
from os import unlink, rmdir
from tempfile import mkdtemp
from time import time, sleep, gmtime
import subprocess
import re
import random
from functools import partial
import logging
import operator

import requests

sys.path.insert(0, dirname(dirname(dirname(abspath(__file__)))))
import ttv
import ttv.config

class TestRateLimitFilter(unittest.TestCase):

    def setUp(self):
        logging.basicConfig(level=logging.INFO)
        
    def test_ratelimitfilter(self):
        logger = logging.getLogger('test_ttv_logger')
        logger.addFilter(ttv.RateLimitFilter(count=2, window=0.5))
        print
        for i in range(0, 9):  # pylint: disable=W0612
            sleep(0.1)
            logger.info('this should print twice, then twice again')
        
        self.assertTrue(True)

    def test_smtphandler(self):
        # hmm
        self.assertTrue(True)


class TestMonotoneSequence(unittest.TestCase):
    
    def test_gms(self):
        self.assertEqual(ttv.greatest_monotone_sequence([1, 2, 3, 2, 4, 2], operator.gt),
                         (4, 5))
        self.assertEqual(ttv.greatest_monotone_sequence([1, 2, 4, 2, 3], operator.gt),
                         (2, 3))
        self.assertEqual(ttv.greatest_monotone_sequence([1, 2, 4, 2, 3], operator.lt),
                         (0, 2))


class TestUniquify(unittest.TestCase):

    def test_uniquify(self):
        self.assertEqual(sorted(ttv.uniquify([0, 1, 2, 2])), 
                         sorted([0, 1, 2]))


class TestChunkify(unittest.TestCase):

    def test_chunkify(self):
        self.assertEqual(list(ttv.chunkify([0, 1, 2, 3, 4, 5], 2)),
                         [(0, 1), (2, 3), (4, 5)])
        self.assertEqual(list(ttv.chunkify(xrange(0, 6), 3)),
                         [(0, 1, 2), (3, 4, 5)])
        self.assertEqual(list(ttv.chunkify([0, 1, 2, 3, 4, 5], 4)),
                         [(0, 1, 2, 3), (4, 5)])


class TestHybridSorted(unittest.TestCase):

    def test_hsk(self):
        self.assertEqual(sorted(['app12', 'app1', 'app8'], key=ttv.hsk),
                         ['app1', 'app8', 'app12'])


class TestAutoremove(unittest.TestCase):

    def setUp(self):
        self.tmpdir = mkdtemp()  # pylint: disable=W0201

    def tearDown(self):
        if isfile(self.tmpfile):
            unlink(self.tmpfile)
        rmdir(self.tmpdir)

    def test_autoremove(self):
        def filemaker():
            self.tmpfile = os.path.join(self.tmpdir, 'foofile')  # pylint: disable=W0201
            h = open(self.tmpfile, 'w')
            h.write('foo')
            h.flush()
            return self.tmpfile
        
        with ttv.autoremove(filemaker()) as f:
            self.assertTrue(isfile(f))
        self.assertFalse(isfile(f))


class TestOSFuncIgnoreEtc(unittest.TestCase):
    
    def setUp(self):
        self.tmpdir = mkdtemp()  # pylint: disable=W0201

    def tearDown(self):
        rmdir(self.tmpdir)

    def test_unlink_ignore(self):
        foofile = os.path.join(self.tmpdir, 'foo')
        open(foofile, 'w').write('foo')
        self.assertTrue(isfile(foofile))

        ttv.unlink_ignore(foofile)
        self.assertFalse(isfile(foofile))

        with self.assertRaises(OSError):
            unlink(foofile)
        ttv.unlink_ignore(foofile)
        

class TestPercentFull(unittest.TestCase):
    
    def test_percent_full(self):
        
        df = subprocess.check_output(['df', '-k', '/'])
        m = re.search(r'\s(\d+)%', df)
        self.assertTrue(m)
        dfpct = int(m.group(1))
        pkgpct = ttv.percent_full('/')
        self.assertTrue(dfpct - 1 <= pkgpct <= dfpct + 1)


class TestRequestsWrapper(unittest.TestCase):
    # pylint: disable=E1103
    
    @ttv.timeout(4)
    def test_requests_wrapper(self):
        print
        r = ttv.requests_wrapper(requests.get, 'http://www.google.com/')
        self.assertTrue(r is not None)
        self.assertTrue(r.ok)
        r = ttv.requests_wrapper(requests.get, 'http://definitely.wont.work.com/', quiet=True)
        self.assertTrue(r is None)
            

class TestSUS(unittest.TestCase):

    @ttv.timeout(4)
    def test_sleep_until_secs(self):
        now = gmtime()[5]
        print '\nsleeping for a couple of seconds...'
        ttv.sleep_until_secs([(now + 2) % 60, (now - 2) % 60])
        print 'awake now'
        self.assertTrue(True)

    def test_sleep_until_secs_more(self):
        secs = gmtime()[5]
        secs_list = range(secs - 1, secs + 5)
        
        start = time()
        print
        for _ in (0, 1, 2):
            print 'this message will print once per second...no more and no less'
            ttv.sleep_until_secs(secs_list)
        et = time() - start
        self.assertTrue(2 < et < 4)


class TestThrottle(unittest.TestCase):
    def test_throttle(self):
        t = ttv.Throttle()

        # it shouldn't be under, and it should be over by only a little
        mark = time()
        t.throttle(0.2)
        self.assertTrue(mark + 0.2 <= time() <= mark + 0.4)
        mark = time()
        t.throttle(0.4)
        self.assertTrue(mark + 0.4 <= time() <= mark + 0.6)


class TestSpinnerThrottle(unittest.TestCase):
    def test_spinner(self):  # pylint: disable=R0201
        
        spinsecs = 1.5
        print '\nspinner should spin for %s seconds:  ' % spinsecs,
        spinner = ttv.Spinner(0.10)
        throttle = ttv.Throttle()
        done = time() + spinsecs
        while time() < done:
            if spinner.click():
                sys.stdout.write(spinner.phase())
                sys.stdout.flush()
            throttle.throttle(0.05)
        print


class TestDither(unittest.TestCase):
    def test_dither(self):
        f = ttv.dither(100)
        for _ in range(1, 100):
            val = f()
            self.assertTrue(90 <= val <= 110)


@ttv.timeout(1.0)
def sleeper(secs):
    sleep(secs)

class TestTimeout(unittest.TestCase):
    def test_timeout(self):
        sleeper(0.5)
        
        try:
            sleeper(2.0)
        except ttv.TimeoutException:
            pass
        else:
            logging.error('timeout malfunctioned!')
            self.assertTrue(False)


class TestTimelimit(unittest.TestCase):
    def test_timelimit(self):

        with ttv.timelimit(0.3):
            sleep(0.1)
        
        timeout = False
        try:
            with ttv.timelimit(0.1):
                sleep(0.3)
        except ttv.TimeoutException:
            timeout = True
        self.assertTrue(timeout)


@ttv.cached(0.2)
def fff(x, y, z='42'):
    return (x, y, z, time())

@ttv.cached(0.2)
def ggg():
    return {'a': 1, 'b': 2}

@ttv.cached(0.2, deepcopy=True)
def hhh():
    return {'c': 3, 'd': 4}

iii_return_none = True
@ttv.cached(0.2, dont_cache_none=True)
def iii():
    """returns 1, None, 1, None, 1..."""
    global iii_return_none
    iii_return_none = not iii_return_none
    return None if iii_return_none else 1

jjj_return_none = True
@ttv.cached(0.2, dont_cache_none=True, use_stale_non_none=True)
def jjj():
    """returns 1, None, 1, None, 1..."""
    global jjj_return_none
    jjj_return_none = not jjj_return_none
    return None if jjj_return_none else 1

class TestCached(unittest.TestCase):

    def test_cached(self):
        v1 = fff(1, 2, z=12)
        v2 = fff(1, 2, z=12)
        self.assertEqual(v1, v2)
        v3 = fff(1, 2, z=13)
        self.assertNotEqual(v1, v3)
        sleep(0.3)
        v4 = fff(1, 2, z=12)
        self.assertNotEqual(v1, v4)

    def test_cached_deepcopy(self):
        self.assertTrue(id(ggg()) == id(ggg()))
        self.assertTrue(id(hhh()) != id(hhh()))
        self.assertTrue(hhh() == hhh())

    def test_cached_flags_1(self):
        self.assertTrue(iii() == 1)
        self.assertTrue(iii() == 1)
        sleep(0.3)
        self.assertTrue(iii() is None)
        self.assertTrue(iii() == 1)

    def test_cached_flags_2(self):
        self.assertTrue(jjj() == 1)
        self.assertTrue(jjj() == 1)
        sleep(0.3)
        self.assertTrue(jjj() == 1)


class TestCachedHttpGetter(unittest.TestCase):
    # pylint: disable=E1103
    def test_chg(self):
        url = 'http://24timezones.com/world_directory/current_san_francisco_time.php'
        get = ttv.cached_http_getter(2.0)
        j1 = get(url).text
        sleep(1.1)
        j2 = get(url).text
        self.assertEqual(j1, j2)
        sleep(2.1)
        j3 = get(url).text
        self.assertNotEqual(j1, j3)


@ttv.objcached(0.5)
class Foo(object):
    def __init__(self):
        self.timestamp = time()


@ttv.objcached(partial(random.uniform, 0.3, 0.7))
class Bar(object):
    def __init__(self):
        self.timestamp = time()


class TestObjcached(unittest.TestCase):
    def test_objcached_with_num(self):
        foo1 = Foo()
        foo2 = Foo()
        self.assertEqual(foo1.timestamp, foo2.timestamp)
        sleep(1.0)
        foo3 = Foo()
        self.assertNotEqual(foo1.timestamp, foo3.timestamp)
        
    def test_objcached_with_func(self):
        bar1 = Bar()
        bar2 = Bar()
        self.assertEqual(bar1.timestamp, bar2.timestamp)
        sleep(1.0)
        bar3 = Bar()
        self.assertNotEqual(bar1.timestamp, bar3.timestamp)


@ttv.borg
class Barf(object):
    def __init__(self):
        self.timestamp = time()


class TestBorg(unittest.TestCase):
    def test_borg(self):
        x = Barf()
        sleep(0.5)
        y = Barf()
        self.assertEqual(x.timestamp, y.timestamp)


class TestJsonCacheFile(unittest.TestCase):
    def setUp(self):
        self.tmpdir = mkdtemp()  # pylint: disable=W0201
        self.tmpfile = os.path.join(self.tmpdir, 'jcf.json')

    def tearDown(self):
        if isfile(self.tmpfile):
            unlink(self.tmpfile)
        rmdir(self.tmpdir)

    def test_jsoncachefile(self):
        jcf = ttv.JsonCacheFile(self.tmpfile)
        self.assertTrue(jcf.read(100) is None)
        data = {'a': 1}
        jcf.write(data)
        self.assertEqual(jcf.read(1.0), data)
        self.assertEqual(jcf.read(), data)
        sleep(0.3)
        self.assertTrue(jcf.read(0.1) is None)
        data = {'b': 2}
        jcf.write(data)
        self.assertEqual(jcf.read(1.0), data)
        

if __name__ == '__main__':
    unittest.main()
