class twitch_rabbit::server (
  $cookie                = $twitch_rabbit::rabbit_erlang_cookie,
  $rabbit_cluster        = $twitch_rabbit::rabbit_cluster,
  $rabbit_conf_dir       = $twitch_rabbit::rabbit_conf_dir,
  $rabbit_cluster_config = $twitch_rabbit::rabbit_cluster_config,
  $consul_service_name   = undef,
  $consul_tags           = [],
  $amqp_port             = $twitch_rabbit::amqp_port,
  $http_api_port         = $twitch_rabbit::http_api_port,
) inherits twitch_rabbit {
  validate_array($rabbit_cluster)

  package{ 'rabbitmq-server':
    ensure=> installed,
  }

  file{ "${rabbit_conf_dir}/rabbitmq-env.conf":
    ensure  => present,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    source  => 'puppet:///modules/twitch_rabbit/rabbitmq.conf',
    require => [Package['rabbitmq-server']],
  }

  exec { '/usr/sbin/rabbitmq-plugins enable rabbitmq_management':
    unless      => "/usr/sbin/rabbitmq-plugins list rabbitmq_management\$ | grep -q '[E]'",
    require     => Package['rabbitmq-server'],
    notify      => Service['rabbitmq-server'],
    environment => ['HOME=/root'], # required for /root/.erlang.cookie
  }

  file { '/var/lib/rabbitmq/.erlang.cookie':
    ensure  => present,
    owner   => 'rabbitmq',
    group   => 'rabbitmq',
    mode    => '0400',
    content => $cookie,
    require => Package['rabbitmq-server'],
  }

  file{ "${rabbit_conf_dir}/rabbitmq.config":
    ensure  => $rabbit_cluster_config,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => template('twitch_rabbit/rabbitmq_cluster_conf.erb'),
    require => [Package['rabbitmq-server']],
  }

  service { 'rabbitmq-server':
    enable  => true,
    ensure  => running,
    require => [
      Package['rabbitmq-server'],
      File['/etc/rabbitmq/rabbitmq-env.conf'],
      File['/var/lib/rabbitmq/.erlang.cookie'],
      File['/etc/rabbitmq/rabbitmq.config'],
    ],
  }

  logrotate::rule{ 'rabbitmq-server':
    ensure        => present,
    path          => '/var/log/rabbitmq/*.log',
    rotate_every  => 'day',
    missingok     => true,
    rotate        => 20,
    compress      => true,
    delaycompress => true,
    ifempty       => false,
    sharedscripts => true,
    postrotate    => '/etc/init.d/rabbitmq-server rotate-logs > /dev/null',
  }

  cron { 'rabbit_stats':
    ensure  => present,
    command => "sudo /usr/sbin/rabbitmqctl list_queues 2>/dev/null | grep -Fv ... | awk '{print \"worker_queue queue=\"$1\" length=\"$2}' | logger -p local1.info -t worker_queue",
    user    => 'jtv',
    require => Package['rabbitmq-server'],
  }

  # consul service config set in hiera/fqdn
  if $consul_service_name {
    consul::service { $consul_service_name:
      port        => $amqp_port,
      consul_tags => $consul_tags
    }

    consul::service { "${consul_service_name}-http-api":
      port        => $http_api_port,
      consul_tags => $consul_tags
    }
  }

}
