module Puppet::Parser::Functions
  newfunction(:validate_ip_private, :doc => <<-ENDHEREDOC
    Validate that all values passed are loopback or private IP addresses.
    Fail compilation if any value fails this check.

    Private IP addresses are defined as part of the networks:
      10.0.0.0/8
      127.0.0.0/8 (loopback)
      172.16.0.0/12
      192.168.0.0/16
  ENDHEREDOC
  ) do |args|
    require "ipaddr"
    rescuable_exceptions = [ ArgumentError ]

    if defined?(IPAddr::InvalidAddressError)
      rescuable_exceptions << IPAddr::InvalidAddressError
    end

    unless args.length > 0 then
      raise Puppet::ParseError, ("validate_ip_private(): wrong number of arguments (#{args.length}; must be > 0)")
    end

    args.flatten.each do |arg|
      unless arg.is_a?(String)
        raise Puppet::ParseError, "#{arg.inspect} is not a string"
      end

      begin
        addr = IPAddr.new(arg)
        if addr.ipv4?
          if not IPAddr.new('10.0.0.0/8').include?(addr) and
            not IPAddr.new('127.0.0.0/8').include?(addr) and
            not IPAddr.new('172.16.0.0/12').include?(addr) and
            not IPAddr.new('192.168.0.0/16').include?(addr)
            raise Puppet::ParseError, "#{arg.inspect} is not a private IP address"
          end
        else
          raise Puppet::ParseError, "#{arg.inspect} is an IPv6 address and is not supported"
        end
      rescue *rescuable_exceptions
        raise Puppet::ParseError, "#{arg.inspect} is not a valid IP address"
      end
    end
  end
end
