# == Define: twitch_redis::server
#
# Configure redis server on an arbitrary port.
#
# === Parameters
#
# [*port*]
#   Accept redis connections on this port.
#   Default: 6379
#
# [*bind*]
#   Listen on a single IP address or array of IP addresses.
#   To avoid publically exposing Redis hosts, we also validate that the IPs are private.
#   NOTE: Redis 3.2+ requires specifying either bind or password.
#   Default: undef, bind to all interfaces if given a port.
#
# [*maxclients*]
#   Set the redis config value maxclients. If no value provided, it is
#   not included in the configuration for 2.6 and set to 0 (unlimited)
#   for 2.4. Defaults to 10000 for 3.0+.
#   Default: 0 (2.4)
#   Default: nil (2.6)
#   Default: 10000 (3.0)
#
# [*timeout*]
#   Set the redis config value timeout (seconds).
#   Default: 0 (no timeout)
#
# [*log_level*]
#   Set the redis config value loglevel. Valid values are debug,
#   verbose, notice, and warning.
#   Default: notice
#
# [*databases*]
#   Set the redis config value databases.
#   Default: 16
#
# [*slowlog_log_slower_than*]
#   Set the redis config value slowlog-log-slower-than (microseconds).
#   Default: 10000
#
# [*slowlog_max_len*]
#   Set the redis config value slowlog-max-len.
#   Default: 1024
#
# [*password*]
#   Password used by AUTH command. Will be setted is its not nil.
#   NOTE: Redis 3.2+ requires specifying either bind or password.
#   Default: nil
#
# [*limit_fd*]
#   File descriptor ulimit.
#   Default: 20000
#
# === Examples
#
# redis::server { 'redis-6900':
#   port       => '6900',
#   redis_max_memory => '64gb',
# }
#
# === Authors
#
# Kevin Mullin
#
define twitch_redis::server (
  $ensure                      = 'present',
  $port                        = '6379',
  $timeout                     = 0,
  $log_level                   = 'notice',
  $databases                   = 16,
  $tcp_keepalive               = 0,
  $save                        = [],
  $usesocket                   = false,
  $unixsocketperm              = 755,
  $manage_service              = false,
  $stop_writes_on_bgsave_error = false,
  $slowlog_log_slower_than     = undef,
  $slowlog_max_len             = undef,
  $password                    = undef,
  $maxmemory                   = undef,
  $maxmemory_policy            = 'volatile-lru',
  $maxclients                  = undef,
  $bind                        = undef,
  $process_manager             = 'monit',
  $cluster_mode                = false,
  $cluster_node_timeout        = 5000,
  $limit_fd                    = 20000,
) {

  # we have to require redis here as its a dependency of a server
  require twitch_redis

  validate_bool($manage_service, $usesocket, $cluster_mode)
  validate_re($maxmemory_policy, ['volatile-lru','allkeys-lru','volatile-random','allkeys-random','volatile-ttl','noeviction'])
  validate_array($save)

  $redis_conf_dir = $twitch_redis::redis_conf_dir
  validate_absolute_path($redis_conf_dir)

  # Ensure we only bind to private IPs so we don't accidentally make Redis externally available.
  if $bind and !$password {
    validate_ip_private($bind)
  }

  $config_file    = "${redis_conf_dir}/redis_${name}.conf"
  $dbfilename     = "redis_${name}.rdb"
  $pidfile        = "/var/run/redis/redis_${name}.pid"

  if $ensure == 'present' {
    # present
    $_file_ensure    = 'present'
    $_service_ensure = 'running'
    $_service_enable = true
  } else {
    # absent
    $_file_ensure    = 'absent'
    $_service_ensure = 'stopped'
    $_service_enable = false
  }

  $exec_cmd = join([
    "/sbin/start-stop-daemon --start",
    "--chuid redis:redis",
    "--pidfile $pidfile",
    "--umask 007 --exec /usr/bin/redis-server -- $config_file"
  ], ' ')

  if !str2bool($::systemd) {
    upstart::job { "redis_$name":
      run_type_service => false,
      respawn          => true,
      expect           => 'fork',
      limit            => {
        'nofile' => "$limit_fd $limit_fd",
      },
      pre_start => "mkdir -p /var/run/redis && chown redis:redis /var/run/redis",
      exec      => $exec_cmd,
    }
  } else {
    twitch_systemd::unit_file { "redis_$name.service":
      content => template("${module_name}/redis.systemd.service.erb"),
    }
  }

  if $process_manager == 'daemontools' {
    $daemonize = false
    daemontools::supervise { "redis_$name":
      ensure         => $ensure,
      fdlimit        => 262144,
      daemon_dir     => '/var/lib/service',
      manage_service => $manage_service,
      syslog         => 'local3',
      user           => 'redis',
      wd             => '.',
      daemon         => "redis-server ${config_file} 2>&1",
    }
  } elsif $process_manager == 'monit' {
    $daemonize = true
    twitch_monit::check { "redis_$name":
      ensure         => $ensure,
      pidfile        => "/var/run/redis/redis_$name.pid",
      check_interval => 5,
      uid            => 'root',
      start_program  => "/usr/sbin/service redis_$name start",
      stop_program   => "/usr/sbin/service redis_$name stop",
      cwd            => "/var/lib/redis", # this is hardcoded in the ::redis module
    }
  } else {
    fail("only 'monit' and 'daemontools' are currently supported for \$process_manager")
  }

  if $manage_service {
    # only one of these services will be defined at a given time, we use collectors here because
    # they support an 'undef' response if a resource is not found
    Daemontools::Supervise <| title == "redis_$name" |> {
      subscribe => File[$config_file]
    }
    Twitch_monit::Check <| title == "redis_$name" |> {
      subscribe => File[$config_file]
    }
  }

  # below the above if statement because of variables passed into template
  file { $config_file:
    ensure  => $_file_ensure,
    mode    => '0644',
    owner   => 'root',
    group   => 'root',
    content => template("${module_name}/redis.conf.erb")
  }
}
