# define: twitch_sandstorm_agent::resource
#
# Manages a non-file Sandstorm resource, such as a database user's password.
#
# This class is a wrapper around the sandstorm_resource type, which will take
# care of all of the boilerplate required to get the type working smoothly.
#
# In essence, you just tell this class how to manage your resource and it'll
# take care of the rest. In all commands, the string %q will be substitued with
# the plaintext of the secret before the command is run. The raw command will
# never be leaked into puppet logs unless the underlying command outputs its
# raw arguments for some reason.
#
# The commands called look like this, with "true" being exit status 0, and
# "false" being a non-zero exit status. It's mostly intuitive, but a graph might
# help some people:
#
# exists_command -true-> ensure                       / false --> update_command
#                \       -present  -> in_sync_command - true -> done
#                \       -absent  -> destroy_command -> done
#                 \
#                 false-> ensure-absent -> done
#                       > ensure-present -> create_command -> done
# Example:
#
#     twitch_sandstorm_agent::resource { "rabbitmq_user_${username}":
#       ensure          => 'present',
#       key             => "systems/somecluster/production/rabbitmq-${username}",
#       exists_command  => "/usr/sbin/rabbitmqctl -q list_users | grep '^${username}'",
#       create_command  => "/usr/sbin/rabbitmqctl add_user ${username} %q",
#       destroy_command => "/usr/sbin/rabbitmqctl delete_user ${username}",
#       in_sync_command => "/usr/sbin/rabbitmqctl authenticate_user ${username} %q",
#       update_command  => "/usr/sbin/rabbitmqctl change_password ${username} %q",
#       require         => Service['rabbitmq-server'],
#     }
#
# Parameters:
#   *ensure*          - 'present' or 'absent'
#   *key*             - The sandstorm key that contains the secret you want to use.
#
#   *exists_command*  - The command to run to establish whether the resource to
#                       be managed exists. An exit status of 0 means that the
#                       resource exists, while nonzero means it does not.
#                       NOTE: SECRET WILL NOT BE INTERPOLATED INTO THIS COMMAND
#
#   *create_command*  - The command to run to create the resource, which will
#                       happen the resource doesn't exist and ensure=present.
#                       This command is expected to return 0 for success.
#
#   *destroy_command* - The command to run to destroy the resource, which will
#                       happen when the resource exists and ensure=absent.
#                       NOTE: SECRET WILL NOT BE INTERPOLATED INTO THIS COMMAND
#
#   *in_sync_command* - The command to run to check the status of the resource's
#                       secret. This will normally be an authentication test or
#                       some such. This will happen when the resource exists.
#
#   *update_command*  - The command to run to update the resource's secret. This
#                       will happen when the resource exists but is not in_sync.

define twitch_sandstorm_agent::resource (
  $key,

  $exists_command,
  $create_command,
  $destroy_command,
  $in_sync_command,
  $update_command,

  $cwd           = undef,
  $user          = undef,
  $group         = undef,
  $ensure        = 'present',
  $templates_dir = '/var/lib/sandstorm-agent',
) {
  validate_string($key)
  validate_re($ensure, ['^present$', '^absent$'])

  $_scripts_dir = "${templates_dir}/scripts"
  $_secrets_dir = "${templates_dir}/secrets"
  $_skel_dirs = [$templates_dir, $_scripts_dir, $_secrets_dir]

  ensure_resource('file', $_skel_dirs, {
    'ensure' => 'directory',
    'mode'   => '0700',
    'owner'  => 'root',
    'group'  => 'root',
  })

  $_update_script = "${_scripts_dir}/update-${name}.sh"
  $_secret_file = "${_secrets_dir}/resource-${name}.sandstorm"

  file { $_update_script:
    ensure  => $ensure,
    owner   => 'root',
    group   => 'root',
    mode    => '0700',
    content => template("${module_name}/resource-update.sh.erb"),
  }

  $_template_name = "resource-${name}"

  twitch_sandstorm_agent::template { $_template_name:
    ensure          => $ensure,
    destination     => $_secret_file,
    key             => $key,
    owner           => 'root',
    group           => 'root',
    mode            => '0600',
    restart_command => $_update_script,
    delete_dest     => true,
  }

  sandstorm_resource { $name:
    ensure          => $ensure,
    secret_file     => $_secret_file,
    exists_command  => $exists_command,
    create_command  => $create_command,
    destroy_command => $destroy_command,
    in_sync_command => $in_sync_command,
    update_command  => $_update_script,
    cwd             => $cwd,
    user            => $user,
    group           => $group,
    require         => [
      Twitch_sandstorm_agent::Template[$_template_name],
    ],
  }

}
