# define: twitch_sandstorm_agent::template_composite
#
# Creates a blank file that can be composited into with template_fragments
#
# Parameters:
#   [*ensure*]          - "present", "absent", state of managed resources
#   [*destination*]     - The path where the final file should be placed
#   [*restart_command*] - Shell command to run which restarts the service that
#                         depends on the destination file.
#   [*delete_dest*]     - Boolean, whether or not to delete the destination
#                         file when ensure = absent on this resource. Has no
#                         effect when ensure = present. Defaults to false.
#
# Usage:
#  twitch_sandstorm_agent::template_composite { 'wildcard-jtv-cert':
#    destination => '/tmp/test-cert',
#  }
#
#  twitch_sandstorm_agent::template_composite { 'wildcard-jtv-cert':
#    destination => '/tmp/test-cert',
#    owner       => 'custom_user',
#    group       => 'custom_group',
#    mode        => '0600'        # only 'custom_user' can read/write the file
#  }
#

define twitch_sandstorm_agent::template_composite (
  $destination = '',
  $restart_command = '',
  $ensure = 'present',
  $owner = 'root',
  $group = 'root',
  $trim_newline = false,
  $mode  = '0644',
  $delete_dest = false,
) {

  include twitch_sandstorm_agent

  $template_source = "${twitch_sandstorm_agent::params::templates_dir}/${name}"
  $config_require = Concat[$template_source]

  $_exec_name = "sandstorm run ${name}"
  $_exec_cmd = "${twitch_sandstorm_agent::sandstorm_run_cmd} --template ${template_source}"
  exec { $_exec_name:
    command     => $_exec_cmd,
    cwd         => '/root',
    environment => ['HOME=/root'],
    refreshonly => true,
    subscribe   => [
      Package[$twitch_sandstorm_agent::package_name],
    ],
    notify      => [
      Class['twitch_sandstorm_agent::service'],
    ],
  }

  concat { $template_source:
    ensure => $ensure,
    notify => Exec[$_exec_name],
  }

  file { "${twitch_sandstorm_agent::confd_dir}/10-${name}.conf":
    ensure  => $ensure,
    content => template("${module_name}/template.erb"),
    notify  => [
      Exec[$_exec_name],
    ],
    require => $config_require,
  }

  if $ensure == 'absent' and $delete_dest {
    file { $destination:
      ensure => 'absent',
      # Make sure the file isn't bucketed by puppet
      backup => false,
    }
  }
}
