# Class: twitch_smtp::flat_relay
# 
# Configures smtp relay for the flat mail infrastructure. Historically Twitch uses multi tier mail infrastructure
# but it is not heavily used any more so it can be simplified.
# 
# To use this class please specify
# twitch_smtp::type: 'flat_relay'
# in hiera settings
#
class twitch_smtp::flat_relay {
  require twitch_smtp::flat_relay::setup
  include twitch_smtp::service
  include twitch_smtp::anycast
  require twitch::pylock
  include twitch_vars

  $twitch_networks = $::twitch_vars::twitch_networks

  package { 'postfix-pcre':
    ensure => present,
  }

  file { '/etc/postfix/main.cf':
    ensure  => file,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => template('twitch_smtp/postfix-main-server.cf.erb'),
    notify  => Class['twitch_smtp::service'],
  }

  file { '/etc/postfix/master.cf':
    ensure  => file,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => template('twitch_smtp/master-server.cf.erb'),
    notify  => Class['twitch_smtp::service'],
  }

  file { '/etc/postfix/recipient_canonical':
    ensure => file,
    owner  => 'root',
    group  => 'root',
    mode   => '0644',
    source => 'puppet:///modules/twitch_smtp/recipient_canonical_relay',
    notify => Class['twitch_smtp::service'],
  }

  file { '/etc/postfix/access_regex':
    ensure => absent,
    before => Class['twitch_smtp::service'],
  }

  file { '/etc/postfix/access_pcre':
    ensure => file,
    owner  => 'root',
    group  => 'root',
    mode   => '0644',
    source => 'puppet:///modules/twitch_smtp/access_pcre',
    notify => Class['twitch_smtp::service'],
  }
  $maildir_cron_cmds = [
    'mkdir -p /dev/shm/hardbounces',
    'chown nobody:nogroup /dev/shm/hardbounces',
    '/usr/local/bin/pylock --lock /tmp/process_maildir_bounces.lock -- /usr/local/bin/bounces.py -v -m /dev/shm/hardbounces',
  ]
  $maildir_cron_log = 'logger -p local3.info -t process_maildir_bounces'
  twitch_cron { 'process_maildir_bounces':
    ensure  => present,
    command => join([join($maildir_cron_cmds, ';'), $maildir_cron_log], '|'),
    user    => 'root',
    require => File['/usr/local/bin/bounces.py'],
  }

  file {'/usr/local/bin/bounces.py':
    ensure => file,
    owner  => 'root',
    group  => 'root',
    mode   => '0755',
    source => 'puppet:///modules/twitch_smtp/bounces.py',
  }

  file { '/etc/postfix/scache_destinations':
    ensure => file,
    owner  => 'root',
    group  => 'root',
    mode   => '0644',
    source => 'puppet:///modules/twitch_smtp/scache_destinations',
    notify => Class['twitch_smtp::service'],
  }

  file { '/etc/postfix/transport_pcre':
    ensure  => file,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => '',
    notify  => Class['twitch_smtp::service'],
  }

  file { '/usr/local/bin/postfix_balancer.py':
    ensure  => file,
    owner   => 'root',
    group   => 'root',
    mode    => '0755',
    source  => 'puppet:///modules/twitch_smtp/postfix_balancer.py',
    require => File['/etc/postfix/interfaces.conf'],
  }

  file { '/etc/postfix/interfaces.conf':
    ensure  => file,
    owner   => 'root',
    group   => 'root',
    mode    => '0644',
    content => template('twitch_smtp/interfaces.conf.erb'),
  }

  twitch_systemd::service { 'postfix_balancer':
    command   => '/usr/local/bin/postfix_balancer.py',
    options   => '-c /etc/postfix/interfaces.conf',
    user      => 'nobody',
    restart   => 'always',
    subscribe => File[
      '/usr/local/bin/postfix_balancer.py',
      '/etc/postfix/interfaces.conf'
    ],
  }

  twitch_ssl_certificate::certificate { 'mail-private-crt':
    cert_filesystem_path => '/etc/ssl/certs/smtp-out_justin_tv.crt',
    cert_contents        => '{{ key "vidcs/smtp_out_cert/production/smtp-out_justin_tv/cert" }}',
    cert_mode            => '0600',
    key_identifier       => 'mail-private-key',
    key_filesystem_path  => '/etc/ssl/certs/smtp-out_justin_tv.key',
    key_contents         => '{{ key "vidcs/smtp_out_cert/production/smtp-out_justin_tv/key" }}',
    key_mode             => '0600',
    notify               => Class['twitch_smtp::service'],
  }

  twitch_ssl_certificate::certificate { 'mail-private-digicert':
    cert_filesystem_path => '/etc/ssl/certs/DigiCertCA.crt',
    cert_contents        => '{{ key "vidcs/smtp_out_cert/production/smtp-out_justin_tv/DigiCertCA" }}',
    cert_mode            => '0600',
    notify               => Class['twitch_smtp::service'],
  }

}
