# This class sets up squid on a squid cache server.
class twitch_squid (
  # Number of cache workers
  $workers                = $twitch_squid::params::workers,
  # On-disk location to store cached files
  $cache_dir              = $twitch_squid::params::cache_dir,
  $cache_refresh_patterns = $twitch_squid::params::cache_refresh_patterns,
  # Enable this to stop tunneling directly out of pop masters or any box
  # without a public ip.
  $no_direct_connect      = $twitch_squid::params::no_direct_connect,
  $egress_local_dc_only   = $twitch_squid::params::egress_local_dc_only,

  $squid_use_syslog       = false,
  $squid_log_loc          = $twitch_squid::params::squid_log_loc,
  $squid_log_format_name  = $twitch_squid::params::squid_log_format_name,
  $squid_syslog_log_fac   = $twitch_squid::params::squid_syslog_log_fac,
  $squid_syslog_log_lev   = $twitch_squid::params::squid_syslog_log_lev,
  $squid_syslog_log_name  = $twitch_squid::params::squid_syslog_log_name,

  $squid_sample           = $twitch_squid::params::squid_sample,
  $squid_sample_log       = $twitch_squid::params::squid_sample_log,
  $squid_sampling_rate    = $twitch_squid::params::squid_sampling_rate,

  $squid_cw               = $twitch_squid::params::squid_cw,

  $squid_version          = $twitch_squid::params::squid_version,
  $consul_tags            = $twitch_squid::params::consul_tags,
  $consul_health_checks   = $twitch_squid::params::consul_health_checks,
  $port                   = $twitch_squid::params::port,
) inherits twitch_squid::params {
  require core

  validate_array($cache_refresh_patterns)
  validate_integer($workers)
  validate_integer($squid_sampling_rate)
  validate_string($squid_log_loc)
  validate_string($squid_syslog_log_fac)
  validate_string($squid_syslog_log_lev)
  validate_string($squid_syslog_log_name)
  validate_string($squid_sample_log)
  validate_bool($no_direct_connect, $squid_use_syslog, $squid_sample, $squid_cw)
  validate_bool($egress_local_dc_only)

  # certain config options require a full restart of the service
  # currently, any change to workers requires a full restart
  # we use a custom fact that detects the current value of workers in the config
  if $::squid_workers != $workers {
    notify { "twitch_squid::config detected workers change (${::squid_workers} -> ${::twitch_squid::workers})":
      before => Class['::twitch_squid::service']
    }
    $_config_notify = Class['::twitch_squid::service']
  } else {
    $_config_notify = Exec['reload-squid']
  }

  exec { 'reload-squid':
    command     => '/bin/systemctl try-reload-or-restart squid',
    refreshonly => true,
  }

  anchor { "${module_name}::start": }
  -> class { "${module_name}::install": }
  -> class { "${module_name}::extras": }
  -> class { "${module_name}::config": }
  -> class { "${module_name}::cache_peers": }
  -> class { "${module_name}::service": }
  -> anchor { "${module_name}::end": }
  Class['twitch_squid::service'] -> Exec['reload-squid']

  twitch_diamond::collector { 'SquidCollector':
    enable => true,
    source => true,
  }

  twitch_diamond::collector { 'ProcessResourcesCollector':
    enable             => true,
    options            => {
      info_keys => 'num_threads,num_fds',
    },
    subsection_options => {
      '[[[process]]]' => {},
      '[[[[squid]]]]' => {
        name => 'squid',
      }
    }
  }

  include twitch_squid::anycast
}
