# Flexible helper that allows a consumer to define an arbitrarily formatted certificate 
# in an arbitrary path to fit their use cases. Places minimal metadata on disk of machine 
# for certzero tooling to discover which certificates are placed on disk.
#
#
# @param ensure
#   Ensure whether your cert config and place should exist or be removed.
#
# @param require_sandstorm
#   Some machine classes don't run sandstorm, and some use-cases only need to place a simple cert file.
#   Setting this option to true simply places a vanilla file resource and not a sandstorm template.
#
# @param cert_filesystem_path
#   The destination path your certificate or PEM container will be placed.
#
# @param cert_contents
#   A sandstorm-supported template string representing your certificate or cert container.
#
# @param cert_owner
#   Unix owner of cert file.
#
# @param cert_group
#   Unix group of cert file.
#
# @param cert_mode
#   Unix mode of cert file.
#
# @param cert_restart_cmd
#   Arbitrary shell command when sandstorm updates a certificate's contents.
#
# @param cert_intermediate_path 
#   Monitoring metadata: The path to the intermediate cert for the certificate.
#
# @param cert_san_pattern
#   Monitoring metadata: If the cert has a SAN, the SAN pattern. Used exclusively for the nagios check config.
#
# @param cert_self_signed 
#   Monitoring metadata: Where the certificate is self-signed.
#
# @param cert_ssl_host
#   Monitoring metadata: The name this SSL certificate is for. Used exclusively for the nagios check config.
#
# @param cert_ssl_port
#   Monitoring metadata: The port to run the SSL check against. Used exclusively for the nagios check config.
#
# @param key_path
#   If you keep your key in a separate file than your certificate container, the destination path
#   for the secret key of the certificate.
#
# @param key_contents
#   A sandstorm-supported template string representing the key to your certificate.
#   This value is optional as the certficate you may place could be a standalone cert (e.g: CA cert)
#
# @param key_owner
#   Unix owner of key file.
#
# @param key_group
#   Unix group of key file.
#
# @param key_mode
#   Unix mode of key file.
#
# @param key_restart_cmd
#   Arbitrary shell command when sandstorm updates a key file's contents.
#
# @param monitoring_use_nagios
#   Monitoring metadata: Whether to configure nagios to check your certificate as well.
#
# @param monitoring_check_fs_cert
#   Monitoring metadata: For the nagios check, Check the cert placed with the cert_filesystem_path argument
#   to check the validity/expiration date of the certificate.
#
# @param monitoring_add_opts
#   Monitoring metadata: For the nagios check, an arbitrary list of pipe (|) separated metadata pieces. 
#   Please align with options here: https://github.com/matteocorti/check_ssl_cert#usage,
#   which should provide exhaustive SSL metadata options.
#
# ==============================================================================
# "Migration" params: Please do not use these parameters unless you are migrating an existing sandstorm resource
# to use the wrapper. Please consult #video-coreservices if usage of these parameters are necessary for your migration.
#
# @param cert_identifier
#   Unique identifier for your certificate. Only uses to label puppet resources. Defaults to name.
#
# @param key_identifier
#   Unique identifier for your secret key. Only uses to label puppet resources. Default to ${name}-key
#
define twitch_ssl_certificate::certificate(
  $ensure                   = 'present',
  $require_sandstorm        = true,
  $cert_filesystem_path     = undef,
  $cert_contents            = undef,
  $cert_owner               = 'root',
  $cert_group               = 'root',
  $cert_mode                = '0644',
  $cert_restart_cmd         = undef,
  $cert_intermediate_path   = undef,
  $cert_san_pattern         = undef,
  $cert_self_signed         = false,
  $cert_ssl_host            = 'localhost',
  $cert_ssl_port            = 443,
  $key_filesystem_path      = '/etc/ssl/certs',
  $key_contents             = undef,
  $key_owner                = 'root',
  $key_group                = 'root',
  $key_mode                 = '0640',
  $key_restart_cmd          = undef,
  $monitoring_use_nagios    = true,
  $monitoring_check_fs_cert = true,
  $monitoring_add_opts      = [],
  $cert_identifier          = $name,
  $key_identifier           = "${name}-key",
) {
  validate_absolute_path($cert_filesystem_path)
  validate_string($cert_ssl_host, $cert_identifier)
  validate_integer($cert_ssl_port)
  validate_bool($cert_self_signed)

  unless $::packer_build_name {

    require twitch_ssl_certificate

    if $cert_filesystem_path == undef {
      fail("twitch_ssl_certificate::certificate ${name} requires ${cert_filesystem_path} to be specified.")
    }

    if $key_filesystem_path != undef {
      validate_absolute_path($key_filesystem_path)
    }

    # Set config path
    $conf_file         = $twitch_ssl_certificate::cert_metadata_path
    $certzero_conf_dir = $twitch_ssl_certificate::certzero_conf_dir

    $ensure_boolean = $ensure ? {
      'present' => true,
      default   => false,
    }

    if $require_sandstorm {
      twitch_sandstorm_agent::template { $cert_identifier:
        ensure      => $ensure,
        destination => $cert_filesystem_path,
        contents    => $cert_contents,
        owner       => $cert_owner,
        group       => $cert_group,
        mode        => $cert_mode,
      }

      # It's a terrible idea to not notify a service if something updates, but in case something doesn't
      # warrant a restart, then allow that.
      if $ensure_boolean and ($cert_restart_cmd != undef) {
        Twitch_sandstorm_agent::Template[$cert_identifier] {
          restart_command => $cert_restart_cmd,
        }
      }

      # If the key is placed separated from the cert
      if ($key_contents != undef) {
        # Write the key
        twitch_sandstorm_agent::template { $key_identifier:
          ensure      => $ensure,
          destination => $key_filesystem_path,
          contents    => $key_contents,
          owner       => $key_owner,
          group       => $key_group,
          mode        => $key_mode,
        }
        # It's a terrible idea to not notify a service if something updates, but in case something doesn't
        # warrant a restart, then allow that.
        if $ensure_boolean and ($key_restart_cmd != undef) {
          Twitch_sandstorm_agent::Template[$key_identifier] {
            restart_command => $key_restart_cmd,
          }
        }
      }
    } else {
      # This is only for machine classes not running sandstorm that only want to place public certs via puppet.
      # Example usecase is puppet CA certs placed on all machines.
      file { $cert_filesystem_path:
        ensure  => $ensure,
        owner   => $cert_owner,
        group   => $cert_group,
        mode    => $cert_mode,
        content => $cert_contents,
      }
    }

    # Configs for monitoring check

    # Configure the nagios monitoring check if specified.
    if $monitoring_use_nagios {

      if($cert_san_pattern != undef){
        validate_string($cert_san_pattern)
      }

      if($cert_intermediate_path != undef){
        validate_absolute_path($cert_intermediate_path)
      }

      $_monitoring_check_fs_cert = $monitoring_check_fs_cert ? {
        true  => "-f|${cert_filesystem_path}",
        false => '',
      }

      $_service_name = $cert_san_pattern ? {
        undef   => '',
        default => "|-n|${cert_san_pattern}|--altnames"
      }

      $_self_signed = $cert_self_signed ? {
        true    => '|-s',
        default => '',
      }

      $_intermediate_certs = $cert_intermediate_path ? {
        undef   => '',
        default => "|-r|${cert_intermediate_path}",
      }


      if(! empty($monitoring_add_opts)){
        $_options = join(['|',$monitoring_add_opts],'|')
      }
      else {
        $_options = ''
      }

      # concat::fragment's ensure key has no effect.
      if $ensure == 'present' {
        concat::fragment { "check_${name}${cert_ssl_host}_${cert_ssl_port}":
          target  => $conf_file,
          content => "-H|${cert_ssl_host}|-p|${cert_ssl_port}|${_monitoring_check_fs_cert}${_service_name}${_self_signed}${_intermediate_certs}${_options}",
          require => Concat[$conf_file],
        }
      }
    }

    # form certzero config blob and place config file
    $certzero_config = delete_undef_values({
      'id'                => $name,
      'cert_path'         => $cert_filesystem_path,
      'cert_self_signed'  => $cert_self_signed,
      'intermediate_path' => $cert_intermediate_path,
      'key_path'          => $key_filesystem_path,
      })

      # perms associated with root instead of specified from the defined resource since certzero tooling
      # should be the only thing to interact with this config.
      file { "${certzero_conf_dir}/cert_${name}.json":
        ensure  => $ensure,
        mode    => '0644',
        owner   => 'root',
        group   => 'root',
        content => pretty_json($certzero_config),
      }

  }

}
