# == Class twitch_ssl_certificate::config
#
# Configures certzero directory and metadata. 
# Includes config 
class twitch_ssl_certificate::config  (
  $ensure              = $twitch_ssl_certificate::params::ensure,
  $certzero_dir        = $twitch_ssl_certificate::params::certzero_dir,
  $certzero_conf_dir   = $twitch_ssl_certificate::params::certzero_conf_dir,
  $certzero_bin_dir    = $twitch_ssl_certificate::params::certzero_bin_dir,
  $enable_nagios_check = $twitch_ssl_certificate::params::enable_nagios_check,
  $cert_metadata_path  = $twitch_ssl_certificate::params::cert_metadata_path,
  $check_interval      = $twitch_ssl_certificate::params::check_interval,
  $check_retry         = $twitch_ssl_certificate::params::check_retry,
  $check_global_opts   = $twitch_ssl_certificate::params::check_global_opts,
  $owner               = $twitch_ssl_certificate::params::owner,
  $group               = $twitch_ssl_certificate::params::group,
) inherits twitch_ssl_certificate::params {

  validate_absolute_path($cert_metadata_path, $certzero_dir, $certzero_conf_dir, $certzero_bin_dir)
  validate_string($check_global_opts, $owner, $group)
  validate_integer($check_interval)
  validate_integer($check_retry)
  validate_bool($enable_nagios_check)

  # wrapper config
  $check_things   = 'certs'

  $certzero_dir_ensure = $ensure ? {
    'present'=> directory,
    default  => absent,
  }

  # ensure for the puppet add-ons
  $nagios_file_ensure = $enable_nagios_check ? {
    true    => file,
    false   => absent,
    default => file,
  }

  $nagios_ensure = $enable_nagios_check ? {
    true    => present,
    default => absent,
  }

  # certzero top dir
  file { $certzero_dir:
    ensure  => $certzero_dir_ensure,
    owner   => $owner,
    group   => $group,
    mode    => '0755',
  }

  # certzero conf dir
  file { $certzero_conf_dir:
    ensure  => $certzero_dir_ensure,
    owner   => $owner,
    group   => $group,
    mode    => '0755',
    purge   => true,
    require => File[$certzero_dir],
  }

  # certzero bin dir
  file { $certzero_bin_dir:
    ensure  => $certzero_dir_ensure,
    owner   => $owner,
    group   => $group,
    mode    => '0755',
    require => File[$certzero_dir],
  }

  # Place the OSS check scripts and wrapper below
  $check_path   = "${certzero_bin_dir}/check_ssl_cert"
  $wrapper_path = "${certzero_bin_dir}/check_ssl_cert_wrapper"
  # https://github.com/matteocorti/check_ssl_cert
  file { $check_path:
    ensure  => $nagios_file_ensure,
    owner   => $owner,
    group   => $group,
    source  => "puppet:///modules/${module_name}/check_ssl_cert",
    require => File[$certzero_bin_dir],
  }

  file { $wrapper_path:
    ensure  => $nagios_file_ensure,
    owner   => $owner,
    group   => $group,
    mode    => '0755',
    source  => "puppet:///modules/${module_name}/check_ssl_cert_wrapper",
    require => File[$certzero_bin_dir],
  }

  # Nagios check config: Short term until certzero CW metrics in prod.
  # Not wrapped in ensure because checks directory is recursively managed by puppet
  if $enable_nagios_check {
    twitch_servicecheck::passive { 'SSL_Validation':
      command  => "${wrapper_path} ${cert_metadata_path} ${check_global_opts}",
      interval => $check_interval,
      retry    => $check_retry,
    }
  }

  # config file for nagios checking
  concat { $cert_metadata_path:
    ensure         => $nagios_ensure,
    owner          => $owner,
    group          => $group,
    mode           => '0644',
    ensure_newline => true,
  }

}
