# == Class: twitch_ssm_tools
#
# Installs the ssm-tools tooling to drive AWS SSM registration for Twitch Bare Metal nodes
#
# === Parameters
#
# [*package_ensure*]
#  Value passed to the ensure parameter of the ssm-tools package, Default: latest
#
# [*enable_ssm*]
#  Toggle the installation and setup of ssm-tools, Default: false
#
# [*deregister_on_stop*]
#  ssm-tools sets up a systemd service that will deregister when the service stops based 
#   on this setting, Default: false
#
# [*twitch_team_whitelist*]
#  Target servers owned by Video organization, Default: ['video']
#
class twitch_ssm_tools (
  $package_ensure        = $twitch_ssm_tools::params::package_ensure,
  $enable_ssm            = $twitch_ssm_tools::params::enable_ssm,
  $deregister_on_stop    = $twitch_ssm_tools::params::deregister_on_stop,
  $twitch_team_whitelist = $twitch_ssm_tools::params::twitch_team_whitelist,
) inherits twitch_ssm_tools::params {

  # Requirements:
  #  - SystemD
  #  - twitch_team is set and part of twitch_team_whitelist
  #  - enable_ssm is true
  #  - node is bare metal
  if $::virtual == 'physical' and str2bool($enable_ssm) and member($twitch_team_whitelist,$::twitch_team) and str2bool($::systemd) {
    $_deregister_on_stop = str2bool($deregister_on_stop)

    package{ 'ssm-tools':
      ensure =>  $package_ensure,
    }

    file{ '/etc/default/ssm_manager':
      ensure  => present,
      owner   => 'root',
      group   => 'root',
      mode    => '0644',
      content => template("${module_name}/etc/default/ssm_manager.erb"),
      require => Package['ssm-tools']
    }

    # Ensure we generate metrics from ssm_tools
    # The ssm-tools package will enable the collector
    # but puppet will purge any collector not configured by it
    twitch_diamond::collector{ 'SSMCollector':
      enable  => true,
      require => Package['ssm-tools']
    }
  }
}
